/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.google.sheets;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import com.google.api.services.sheets.v4.model.Sheet;
import com.google.api.services.sheets.v4.model.SheetProperties;
import com.google.api.services.sheets.v4.model.Spreadsheet;
import com.google.api.services.sheets.v4.model.SpreadsheetProperties;
import com.google.api.services.sheets.v4.model.ValueRange;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.component.google.sheets.internal.GoogleSheetsConstants;
import org.apache.camel.component.google.sheets.server.GoogleSheetsApiTestServer;
import org.apache.camel.component.google.sheets.server.GoogleSheetsApiTestServerRule;
import org.apache.camel.support.PropertyBindingSupport;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.extension.RegisterExtension;

/**
 * Abstract base class for GoogleSheets Integration tests generated by Camel API component maven plugin.
 */
@TestInstance(TestInstance.Lifecycle.PER_METHOD)
public class AbstractGoogleSheetsTestSupport extends CamelTestSupport {

    protected static final String TEST_SHEET = "TestData";
    private static final String TEST_OPTIONS_PROPERTIES = "/test-options.properties";

    @RegisterExtension
    protected GoogleSheetsApiTestServerRule googleSheetsApiTestServerRule
            = new GoogleSheetsApiTestServerRule(TEST_OPTIONS_PROPERTIES);

    private Spreadsheet spreadsheet;

    private static Properties loadProperties() {
        // read GoogleMail component configuration from TEST_OPTIONS_PROPERTIES
        final Properties properties = new Properties();
        try {
            properties.load(AbstractGoogleSheetsTestSupport.class.getResourceAsStream(TEST_OPTIONS_PROPERTIES));
        } catch (Exception e) {
            throw new RuntimeCamelException(
                    String.format("%s could not be loaded: %s", TEST_OPTIONS_PROPERTIES, e.getMessage()), e);
        }
        return properties;
    }

    // Used by JUnit to determine whether or not to run the integration tests
    @SuppressWarnings("unused")
    private static boolean hasCredentials() {
        Properties properties = loadProperties();

        return !properties.getProperty("clientId", "").isEmpty()
                && !properties.getProperty("clientSecret", "").isEmpty()
                && !properties.getProperty("accessToken", "").isEmpty()
                || !properties.getProperty("serviceAccountKey", "").isEmpty();
    }

    /**
     * Create test spreadsheet that is used throughout all tests.
     */
    private void createTestSpreadsheet() {
        Spreadsheet spreadsheet = new Spreadsheet();
        SpreadsheetProperties spreadsheetProperties = new SpreadsheetProperties();
        spreadsheetProperties.setTitle("camel-sheets-" + new SecureRandom().nextInt(Integer.MAX_VALUE));

        spreadsheet.setProperties(spreadsheetProperties);

        Sheet sheet = new Sheet();
        SheetProperties sheetProperties = new SheetProperties();
        sheetProperties.setTitle(TEST_SHEET);
        sheet.setProperties(sheetProperties);

        spreadsheet.setSheets(Collections.singletonList(sheet));

        this.spreadsheet = requestBody("google-sheets://spreadsheets/create?inBody=content", spreadsheet);
    }

    /**
     * Add some initial test data to test spreadsheet.
     */
    private void createTestData() {
        if (spreadsheet == null) {
            createTestSpreadsheet();
        }

        ValueRange valueRange = new ValueRange();
        valueRange.setValues(Arrays.asList(Arrays.asList("a1", "b1"), Arrays.asList("a2", "b2")));

        final Map<String, Object> headers = new HashMap<>();
        // parameter type is String
        headers.put(GoogleSheetsConstants.PROPERTY_PREFIX + "spreadsheetId", spreadsheet.getSpreadsheetId());
        // parameter type is String
        headers.put(GoogleSheetsConstants.PROPERTY_PREFIX + "range", TEST_SHEET + "!A1:B2");

        // parameter type is String
        headers.put(GoogleSheetsConstants.PROPERTY_PREFIX + "valueInputOption", "USER_ENTERED");

        requestBodyAndHeaders("google-sheets://data/update?inBody=values", valueRange, headers);
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {

        final CamelContext context = super.createCamelContext();

        final Properties properties = loadProperties();

        Map<String, Object> options = new HashMap<>();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            options.put(entry.getKey().toString(), entry.getValue());
        }

        final GoogleSheetsConfiguration configuration = new GoogleSheetsConfiguration();
        PropertyBindingSupport.bindProperties(context, configuration, options);

        // add GoogleSheetsComponent to Camel context and use localhost url
        final GoogleSheetsComponent component = new GoogleSheetsComponent(context);
        component.setConfiguration(configuration);
        context.addComponent("google-sheets", component);

        return context;
    }

    /**
     * Read component configuration from TEST_OPTIONS_PROPERTIES.
     * 
     * @return             Map of component options.
     * @throws IOException when TEST_OPTIONS_PROPERTIES could not be loaded.
     */
    protected Map<String, Object> getTestOptions() throws IOException {
        final Properties properties = new Properties();
        try {
            properties.load(getClass().getResourceAsStream(TEST_OPTIONS_PROPERTIES));
        } catch (Exception e) {
            throw new IOException(String.format("%s could not be loaded: %s", TEST_OPTIONS_PROPERTIES, e.getMessage()), e);
        }

        Map<String, Object> options = new HashMap<>();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            options.put(entry.getKey().toString(), entry.getValue());
        }

        return options;
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBodyAndHeaders(String endpointUri, Object body, Map<String, Object> headers)
            throws CamelExecutionException {
        return (T) template().requestBodyAndHeaders(endpointUri, body, headers);
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBody(String endpoint, Object body) throws CamelExecutionException {
        return (T) template().requestBody(endpoint, body);
    }

    public Spreadsheet getSpreadsheet() {
        if (spreadsheet == null) {
            createTestSpreadsheet();
        }
        return spreadsheet;
    }

    public Spreadsheet applyTestData(Spreadsheet spreadsheet) {
        createTestData();
        return spreadsheet;
    }

    public void setSpreadsheet(Spreadsheet sheet) {
        this.spreadsheet = sheet;
    }

    public GoogleSheetsApiTestServer getGoogleApiTestServer() {
        return googleSheetsApiTestServerRule.getGoogleApiTestServer();
    }
}
