/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.model;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.camel.resume.ResumeStrategy;
import org.apache.camel.spi.Metadata;

/**
 * Resume EIP to support resuming processing from last known offset.
 */
@Metadata(label = "eip,routing")
@XmlRootElement(name = "resumable")
@XmlAccessorType(XmlAccessType.FIELD)
public class ResumableDefinition extends NoOutputDefinition<ResumableDefinition> {

    @XmlTransient
    private ResumeStrategy resumeStrategyBean;

    @XmlAttribute(required = true)
    @Metadata(required = true, javaType = "org.apache.camel.resume.ResumeStrategy")
    private String resumeStrategy;

    @Metadata(label = "advanced", javaType = "org.apache.camel.LoggingLevel", defaultValue = "ERROR",
              enums = "TRACE,DEBUG,INFO,WARN,ERROR,OFF")
    private String loggingLevel;

    @Override
    public String getShortName() {
        return "resumable";
    }

    @Override
    public String getLabel() {
        return "resumable";
    }

    public ResumeStrategy getResumeStrategyBean() {
        return resumeStrategyBean;
    }

    public String getResumeStrategy() {
        return resumeStrategy;
    }

    public void setResumeStrategy(String resumeStrategy) {
        this.resumeStrategy = resumeStrategy;
    }

    public void setResumeStrategy(ResumeStrategy resumeStrategyBean) {
        this.resumeStrategyBean = resumeStrategyBean;
    }

    public String getLoggingLevel() {
        return loggingLevel;
    }

    public void setLoggingLevel(String loggingLevelRef) {
        this.loggingLevel = loggingLevelRef;
    }

    // Fluent API
    // -------------------------------------------------------------------------

    /**
     * Sets the resume strategy to use
     */
    public ResumableDefinition resumeStrategy(String resumeStrategyRef) {
        setResumeStrategy(resumeStrategyRef);
        return this;
    }

    /**
     * Sets the resume strategy to use
     */
    public ResumableDefinition resumeStrategy(String resumeStrategyRef, String loggingLevelRef) {
        setResumeStrategy(resumeStrategyRef);
        setLoggingLevel(loggingLevelRef);
        return this;
    }

    /**
     * Sets the resume strategy to use
     */
    public ResumableDefinition resumeStrategy(ResumeStrategy resumeStrategy) {
        setResumeStrategy(resumeStrategy);
        return this;
    }

    /**
     * Sets the resume strategy to use
     */
    public ResumableDefinition resumeStrategy(ResumeStrategy resumeStrategy, String loggingLevelRef) {
        setResumeStrategy(resumeStrategy);
        setLoggingLevel(loggingLevelRef);
        return this;
    }
}
