/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.cdi.test;

import java.nio.file.Paths;

import javax.inject.Inject;

import org.apache.camel.CamelContext;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.cdi.CdiCamelExtension;
import org.apache.camel.cdi.ImportResource;
import org.apache.camel.cdi.Uri;
import org.apache.camel.spi.ThreadPoolProfile;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.shrinkwrap.api.Archive;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Test;
import org.junit.runner.RunWith;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.matchesPattern;
import static org.hamcrest.Matchers.notNullValue;

@RunWith(Arquillian.class)
@ImportResource("imported-context.xml")
public class XmlThreadPoolProfileTest {

    @Inject
    @Uri("direct:inbound")
    private ProducerTemplate inbound;

    @Deployment
    public static Archive<?> deployment() {
        return ShrinkWrap.create(JavaArchive.class)
                // Camel CDI
                .addPackage(CdiCamelExtension.class.getPackage())
                // Test Camel XML
                .addAsResource(
                        Paths.get("src/test/resources/camel-context-threadPoolProfile.xml").toFile(),
                        "imported-context.xml")
                // Bean archive deployment descriptor
                .addAsManifestResource(EmptyAsset.INSTANCE, "beans.xml");
    }

    @Test
    public void verifyThreadProfile(CamelContext context) {
        ThreadPoolProfile profile = context.getExecutorServiceManager().getThreadPoolProfile("thread-pool-profile");
        assertThat("Thread pool profile is null!", profile, is(notNullValue()));
        assertThat("Thread pool profile is incorrect!", profile.getPoolSize(), is(equalTo(5)));
    }

    @Test
    public void sendMessageToInbound() {
        String body = inbound.requestBody((Object) "message", String.class);

        assertThat("Body is incorrect!", body,
                matchesPattern("^Processed \\[message] with \\[Camel \\(test\\) thread #\\d+ - Profile Threads]$"));
    }
}
