/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.hdfs;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.security.auth.login.Configuration;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.camel.Processor;
import org.apache.camel.component.hdfs.HdfsComponent;
import org.apache.camel.component.hdfs.HdfsConfiguration;
import org.apache.camel.component.hdfs.HdfsEndpoint;
import org.apache.camel.component.hdfs.HdfsFileType;
import org.apache.camel.component.hdfs.HdfsHeader;
import org.apache.camel.component.hdfs.HdfsInfo;
import org.apache.camel.component.hdfs.HdfsInfoFactory;
import org.apache.camel.component.hdfs.HdfsInputStream;
import org.apache.camel.component.hdfs.Holder;
import org.apache.camel.support.ScheduledPollConsumer;
import org.apache.camel.util.IOHelper;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.slf4j.Logger;

public final class HdfsConsumer
extends ScheduledPollConsumer {
    public static final long DEFAULT_CONSUMER_INITIAL_DELAY = 10000L;
    private final HdfsConfiguration config;
    private final StringBuilder hdfsPath;
    private final Processor processor;
    private final ReadWriteLock rwlock = new ReentrantReadWriteLock();

    public HdfsConsumer(HdfsEndpoint endpoint, Processor processor, HdfsConfiguration config) {
        super((Endpoint)endpoint, processor);
        this.config = config;
        this.hdfsPath = config.getFileSystemType().getHdfsPath(config);
        this.processor = processor;
        this.setUseFixedDelay(true);
    }

    public HdfsEndpoint getEndpoint() {
        return (HdfsEndpoint)super.getEndpoint();
    }

    protected void doStart() throws Exception {
        super.doStart();
        if (this.config.isConnectOnStartup()) {
            this.setupHdfs(true);
        }
    }

    private HdfsInfo setupHdfs(boolean onStartup) throws IOException {
        if (onStartup) {
            this.log.info("Connecting to hdfs file-system {}:{}/{} (may take a while if connection is not available)", new Object[]{this.config.getHostName(), this.config.getPort(), this.hdfsPath});
        } else if (this.log.isDebugEnabled()) {
            this.log.debug("Connecting to hdfs file-system {}:{}/{} (may take a while if connection is not available)", new Object[]{this.config.getHostName(), this.config.getPort(), this.hdfsPath});
        }
        HdfsInfo answer = HdfsInfoFactory.newHdfsInfo(this.hdfsPath.toString(), this.config);
        if (onStartup) {
            this.log.info("Connected to hdfs file-system {}:{}/{}", new Object[]{this.config.getHostName(), this.config.getPort(), this.hdfsPath});
        } else if (this.log.isDebugEnabled()) {
            this.log.debug("Connected to hdfs file-system {}:{}/{}", new Object[]{this.config.getHostName(), this.config.getPort(), this.hdfsPath});
        }
        return answer;
    }

    protected int poll() throws Exception {
        Configuration auth = HdfsComponent.getJAASConfiguration();
        try {
            int n = this.doPoll();
            return n;
        }
        finally {
            HdfsComponent.setJAASConfiguration(auth);
        }
    }

    protected int doPoll() throws IOException {
        FileStatus[] fileStatuses;
        HdfsInfo info = this.setupHdfs(false);
        if (info.getFileSystem().isFile(info.getPath())) {
            fileStatuses = info.getFileSystem().globStatus(info.getPath());
        } else {
            Path pattern = info.getPath().suffix("/" + this.config.getPattern());
            class ExcludePathFilter
            implements PathFilter {
                ExcludePathFilter() {
                }

                public boolean accept(Path path) {
                    return !path.toString().endsWith(HdfsConsumer.this.config.getOpenedSuffix()) && !path.toString().endsWith(HdfsConsumer.this.config.getReadSuffix());
                }
            }
            fileStatuses = info.getFileSystem().globStatus(pattern, (PathFilter)new ExcludePathFilter());
        }
        fileStatuses = Optional.ofNullable(fileStatuses).orElse(new FileStatus[0]);
        return this.processFileStatuses(info, fileStatuses);
    }

    private int processFileStatuses(HdfsInfo info, FileStatus[] fileStatuses) {
        AtomicInteger messageCount = new AtomicInteger(0);
        Arrays.stream(fileStatuses).filter(status -> this.normalFileIsDirectoryHasSuccessFile((FileStatus)status, info)).filter(this::hasMatchingOwner).map(this::createInputStream).filter(Objects::nonNull).forEach(hdfsInputStream -> {
            try {
                this.processHdfsInputStream((HdfsInputStream)hdfsInputStream, messageCount, fileStatuses.length);
            }
            finally {
                IOHelper.close((Closeable)hdfsInputStream, (String)"input stream", (Logger)this.log);
            }
        });
        return messageCount.get();
    }

    private void processHdfsInputStream(HdfsInputStream inputStream, AtomicInteger messageCount, int totalFiles) {
        Holder<Object> key = new Holder<Object>();
        Holder<Object> value = new Holder<Object>();
        while (inputStream.next(key, value) >= 0L) {
            Exchange exchange = this.getEndpoint().createExchange();
            Message message = exchange.getIn();
            String fileName = StringUtils.substringAfterLast((String)inputStream.getActualPath(), (String)"/");
            message.setHeader("CamelFileName", (Object)fileName);
            if (key.value != null) {
                message.setHeader(HdfsHeader.KEY.name(), key.value);
            }
            message.setBody(value.value);
            this.log.debug("Processing file {}", (Object)fileName);
            try {
                this.processor.process(exchange);
            }
            catch (Exception e) {
                exchange.setException((Throwable)e);
            }
            if (exchange.getException() != null) {
                this.getExceptionHandler().handleException((Throwable)exchange.getException());
            }
            int count = messageCount.incrementAndGet();
            this.log.debug("Processed [{}] files out of [{}]", (Object)count, (Object)totalFiles);
        }
    }

    private boolean normalFileIsDirectoryHasSuccessFile(FileStatus fileStatus, HdfsInfo info) {
        if (this.config.getFileType().equals((Object)HdfsFileType.NORMAL_FILE) && fileStatus.isDirectory()) {
            try {
                Path successPath = new Path(fileStatus.getPath().toString() + "/_SUCCESS");
                if (!info.getFileSystem().exists(successPath)) {
                    return false;
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return true;
    }

    private boolean hasMatchingOwner(FileStatus fileStatus) {
        if (this.config.getOwner() != null && !this.config.getOwner().equals(fileStatus.getOwner())) {
            if (this.log.isDebugEnabled()) {
                this.log.debug("Skipping file: {} as not matching owner: {}", (Object)fileStatus.getPath(), (Object)this.config.getOwner());
            }
            return false;
        }
        return true;
    }

    private HdfsInputStream createInputStream(FileStatus fileStatus) {
        try {
            this.rwlock.writeLock().lock();
            HdfsInputStream hdfsInputStream = HdfsInputStream.createInputStream(fileStatus.getPath().toString(), this.config);
            return hdfsInputStream;
        }
        finally {
            this.rwlock.writeLock().unlock();
        }
    }
}

