/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.webhook;

import java.util.List;

import org.apache.camel.Endpoint;
import org.apache.camel.Processor;

/**
 * A {@code WebhookCapableEndpoint} is a special kind of endpoint that is able to operate in webhook mode if required.
 * <p>
 * The interface defines methods for registering/unregistering webhooks as well as callback methods for the webhook workflow.
 */
public interface WebhookCapableEndpoint extends Endpoint {

    /**
     * Creates a processor for the webhook that is responsible for receiving all messages sent to the webhook.
     * <p>
     * The processor should distinguish handshakes from standard calls and forward relevant data to
     * the downstream processor after transformation. Data forwarded to the downstream processor should be of the same
     * type as data generated by the corresponding polling consumer.
     *
     * @param next the downstream processor.
     * @return the webhook route processor
     */
    Processor createWebhookHandler(Processor next);

    /**
     * Register the webhook at the remote site using endpoint specific instruction.
     * <p>
     * Additional configuration is injected into the endpoint using the
     * {@link WebhookCapableEndpoint#setWebhookConfiguration(WebhookConfiguration)} method.
     *
     * @throws Exception if something goes wrong during the registration.
     */
    void registerWebhook() throws Exception;

    /**
     * Unregister the webhook at the remote site using endpoint specific instruction.
     * <p>
     * Additional configuration is injected into the endpoint using the
     * {@link WebhookCapableEndpoint#setWebhookConfiguration(WebhookConfiguration)} method.
     *
     * @throws Exception if something goes wrong during the un-registration.
     */
    void unregisterWebhook() throws Exception;

    /**
     * Used by the workflow manager to inject webhook configuration options.
     *
     * @param webhookConfiguration the webhook configuration options.
     */
    void setWebhookConfiguration(WebhookConfiguration webhookConfiguration);

    /**
     * Used by the endpoint to enlist the HTTP methods it's able to handle.
     * Usually only "POST" is used, but some webhook providers require multiple verbs in the handshake phase.
     *
     * @return the HTTP methods supported by the endpoint
     */
    List<String> getWebhookMethods();

}
