/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.impl;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Set;

import org.apache.camel.BeanInject;
import org.apache.camel.BindToRegistry;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelContextAware;
import org.apache.camel.DeferredContextBinding;
import org.apache.camel.EndpointInject;
import org.apache.camel.NoSuchBeanException;
import org.apache.camel.Produce;
import org.apache.camel.PropertyInject;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.TypeConverter;
import org.apache.camel.spi.CamelBeanPostProcessor;
import org.apache.camel.spi.Registry;
import org.apache.camel.support.DefaultEndpoint;
import org.apache.camel.util.ReflectionHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.camel.support.ObjectHelper.invokeMethod;
import static org.apache.camel.util.ObjectHelper.isEmpty;

/**
 * A bean post processor which implements the <a href="http://camel.apache.org/bean-integration.html">Bean Integration</a>
 * features in Camel. Features such as the <a href="http://camel.apache.org/bean-injection.html">Bean Injection</a> of objects like
 * {@link org.apache.camel.Endpoint} and
 * {@link org.apache.camel.ProducerTemplate} together with support for
 * <a href="http://camel.apache.org/pojo-consuming.html">POJO Consuming</a> via the
 * {@link org.apache.camel.Consume} annotation along with
 * <a href="http://camel.apache.org/pojo-producing.html">POJO Producing</a> via the
 * {@link org.apache.camel.Produce} annotation along with other annotations such as
 * {@link org.apache.camel.DynamicRouter} for creating <a href="http://camel.apache.org/dynamicrouter-annotation.html">a Dynamic router via annotations</a>.
 * {@link org.apache.camel.RecipientList} for creating <a href="http://camel.apache.org/recipientlist-annotation.html">a Recipient List router via annotations</a>.
 * {@link org.apache.camel.RoutingSlip} for creating <a href="http://camel.apache.org/routingslip-annotation.html">a Routing Slip router via annotations</a>.
 * <p/>
 * Components such as <tt>camel-spring</tt>, and <tt>camel-blueprint</tt> can leverage this post processor to hook in Camel
 * bean post processing into their bean processing framework.
 */
public class DefaultCamelBeanPostProcessor implements CamelBeanPostProcessor {

    protected static final Logger LOG = LoggerFactory.getLogger(DefaultCamelBeanPostProcessor.class);
    protected CamelPostProcessorHelper camelPostProcessorHelper;
    protected CamelContext camelContext;

    public DefaultCamelBeanPostProcessor() {
    }

    public DefaultCamelBeanPostProcessor(CamelContext camelContext) {
        this.camelContext = camelContext;
    }

    public Object postProcessBeforeInitialization(Object bean, String beanName) throws Exception {
        LOG.trace("Camel bean processing before initialization for bean: {}", beanName);

        // some beans cannot be post processed at this given time, so we gotta check beforehand
        if (!canPostProcessBean(bean, beanName)) {
            return bean;
        }

        injectClass(bean, beanName);
        injectNestedClasses(bean, beanName);
        injectFields(bean, beanName);
        injectMethods(bean, beanName);

        if (bean instanceof CamelContextAware && canSetCamelContext(bean, beanName)) {
            CamelContextAware contextAware = (CamelContextAware)bean;
            DeferredContextBinding deferredBinding = bean.getClass().getAnnotation(DeferredContextBinding.class);
            CamelContext context = getOrLookupCamelContext();

            if (context == null && deferredBinding == null) {
                LOG.warn("No CamelContext defined yet so cannot inject into bean: {}", beanName);
            } else if (context != null) {
                contextAware.setCamelContext(context);
            }
        }

        return bean;
    }

    public Object postProcessAfterInitialization(Object bean, String beanName) throws Exception {
        LOG.trace("Camel bean processing after initialization for bean: {}", beanName);

        // some beans cannot be post processed at this given time, so we gotta check beforehand
        if (!canPostProcessBean(bean, beanName)) {
            return bean;
        }

        if (bean instanceof DefaultEndpoint) {
            DefaultEndpoint defaultEndpoint = (DefaultEndpoint) bean;
            defaultEndpoint.setEndpointUriIfNotSpecified(beanName);
        }

        return bean;
    }

    /**
     * Strategy to get the {@link CamelContext} to use.
     */
    public CamelContext getOrLookupCamelContext() {
        return camelContext;
    }

    /**
     * Strategy to get the {@link CamelPostProcessorHelper}
     */
    public CamelPostProcessorHelper getPostProcessorHelper() {
        if (camelPostProcessorHelper == null) {
            camelPostProcessorHelper = new CamelPostProcessorHelper(getOrLookupCamelContext());
        }
        return camelPostProcessorHelper;
    }

    protected boolean canPostProcessBean(Object bean, String beanName) {
        return bean != null;
    }

    protected boolean canSetCamelContext(Object bean, String beanName) {
        if (bean instanceof CamelContextAware) {
            CamelContextAware camelContextAware = (CamelContextAware) bean;
            CamelContext context = camelContextAware.getCamelContext();
            if (context != null) {
                LOG.trace("CamelContext already set on bean with id [{}]. Will keep existing CamelContext on bean.", beanName);
                return false;
            }
        }

        return true;
    }

    /**
     * A strategy method to allow implementations to perform some custom JBI
     * based injection of the POJO
     *
     * @param bean the bean to be injected
     */
    protected void injectFields(final Object bean, final String beanName) {
        ReflectionHelper.doWithFields(bean.getClass(), new ReflectionHelper.FieldCallback() {
            public void doWith(Field field) throws IllegalArgumentException, IllegalAccessException {
                PropertyInject propertyInject = field.getAnnotation(PropertyInject.class);
                if (propertyInject != null && getPostProcessorHelper().matchContext(propertyInject.context())) {
                    injectFieldProperty(field, propertyInject.value(), propertyInject.defaultValue(), bean, beanName);
                }

                BeanInject beanInject = field.getAnnotation(BeanInject.class);
                if (beanInject != null && getPostProcessorHelper().matchContext(beanInject.context())) {
                    injectFieldBean(field, beanInject.value(), bean, beanName);
                }

                EndpointInject endpointInject = field.getAnnotation(EndpointInject.class);
                if (endpointInject != null && getPostProcessorHelper().matchContext(endpointInject.context())) {
                    String uri = endpointInject.value().isEmpty() ? endpointInject.uri() : endpointInject.value();
                    injectField(field, uri, endpointInject.property(), bean, beanName);
                }

                Produce produce = field.getAnnotation(Produce.class);
                if (produce != null && getPostProcessorHelper().matchContext(produce.context())) {
                    String uri = produce.value().isEmpty() ? produce.uri() : produce.value();
                    injectField(field, uri, produce.property(), bean, beanName, produce.binding());
                }

                BindToRegistry bind = field.getAnnotation(BindToRegistry.class);
                if (bind != null && getPostProcessorHelper().matchContext(bind.context())) {
                    bindToRegistry(field, bind.value(), bean, beanName);
                }
            }
        });
    }

    public void injectField(Field field, String endpointUri, String endpointProperty,
                               Object bean, String beanName) {
        injectField(field, endpointUri, endpointProperty, bean, beanName, true);
    }

    public void injectField(Field field, String endpointUri, String endpointProperty,
                               Object bean, String beanName, boolean binding) {
        ReflectionHelper.setField(field, bean,
                getPostProcessorHelper().getInjectionValue(field.getType(), endpointUri, endpointProperty,
                        field.getName(), bean, beanName, binding));
    }

    public void injectFieldBean(Field field, String name, Object bean, String beanName) {
        ReflectionHelper.setField(field, bean,
                getPostProcessorHelper().getInjectionBeanValue(field.getType(), name));
    }

    public void injectFieldProperty(Field field, String propertyName, String propertyDefaultValue, Object bean, String beanName) {
        ReflectionHelper.setField(field, bean,
                getPostProcessorHelper().getInjectionPropertyValue(field.getType(), propertyName, propertyDefaultValue,
                        field.getName(), bean, beanName));
    }

    protected void injectMethods(final Object bean, final String beanName) {
        ReflectionHelper.doWithMethods(bean.getClass(), new ReflectionHelper.MethodCallback() {
            public void doWith(Method method) throws IllegalArgumentException, IllegalAccessException {
                setterInjection(method, bean, beanName);
                getPostProcessorHelper().consumerInjection(method, bean, beanName);
            }
        });
    }

    protected void injectClass(final Object bean, final String beanName) {
        Class<?> clazz = bean.getClass();
        BindToRegistry ann = clazz.getAnnotation(BindToRegistry.class);
        if (ann != null && getPostProcessorHelper().matchContext(ann.context())) {
            bindToRegistry(clazz, ann.value(), bean, beanName);
        }
    }

    protected void injectNestedClasses(final Object bean, final String beanName) {
        ReflectionHelper.doWithClasses(bean.getClass(), new ReflectionHelper.ClassCallback() {
            public void doWith(Class clazz) throws IllegalArgumentException, IllegalAccessException {
                BindToRegistry ann = (BindToRegistry) clazz.getAnnotation(BindToRegistry.class);
                if (ann != null && getPostProcessorHelper().matchContext(ann.context())) {
                    // its a nested class so we dont have a bean instance for it
                    bindToRegistry(clazz, ann.value(), null, null);
                }
            }
        });
    }

    protected void setterInjection(Method method, Object bean, String beanName) {
        PropertyInject propertyInject = method.getAnnotation(PropertyInject.class);
        if (propertyInject != null && getPostProcessorHelper().matchContext(propertyInject.context())) {
            setterPropertyInjection(method, propertyInject.value(), propertyInject.defaultValue(), bean, beanName);
        }

        BeanInject beanInject = method.getAnnotation(BeanInject.class);
        if (beanInject != null && getPostProcessorHelper().matchContext(beanInject.context())) {
            setterBeanInjection(method, beanInject.value(), bean, beanName);
        }

        EndpointInject endpointInject = method.getAnnotation(EndpointInject.class);
        if (endpointInject != null && getPostProcessorHelper().matchContext(endpointInject.context())) {
            String uri = endpointInject.value().isEmpty() ? endpointInject.uri() : endpointInject.value();
            setterInjection(method, bean, beanName, uri, endpointInject.property());
        }

        Produce produce = method.getAnnotation(Produce.class);
        if (produce != null && getPostProcessorHelper().matchContext(produce.context())) {
            String uri = produce.value().isEmpty() ? produce.uri() : produce.value();
            setterInjection(method, bean, beanName, uri, produce.property());
        }

        BindToRegistry bind = method.getAnnotation(BindToRegistry.class);
        if (bind != null && getPostProcessorHelper().matchContext(bind.context())) {
            bindToRegistry(method, bind.value(), bean, beanName);
        }
    }

    public void setterInjection(Method method, Object bean, String beanName, String endpointUri, String endpointProperty) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length != 1) {
            LOG.warn("Ignoring badly annotated method for injection due to incorrect number of parameters: {}", method);
        } else {
            String propertyName = org.apache.camel.util.ObjectHelper.getPropertyName(method);
            Object value = getPostProcessorHelper().getInjectionValue(parameterTypes[0], endpointUri, endpointProperty,
                    propertyName, bean, beanName);
            invokeMethod(method, bean, value);
        }
    }

    public void setterPropertyInjection(Method method, String propertyValue, String propertyDefaultValue,
                                        Object bean, String beanName) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length != 1) {
            LOG.warn("Ignoring badly annotated method for injection due to incorrect number of parameters: {}", method);
        } else {
            String propertyName = org.apache.camel.util.ObjectHelper.getPropertyName(method);
            Object value = getPostProcessorHelper().getInjectionPropertyValue(parameterTypes[0], propertyValue, propertyDefaultValue, propertyName, bean, beanName);
            invokeMethod(method, bean, value);
        }
    }

    public void setterBeanInjection(Method method, String name, Object bean, String beanName) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        if (parameterTypes.length != 1) {
            LOG.warn("Ignoring badly annotated method for injection due to incorrect number of parameters: {}", method);
        } else {
            Object value = getPostProcessorHelper().getInjectionBeanValue(parameterTypes[0], name);
            invokeMethod(method, bean, value);
        }
    }

    private void bindToRegistry(Class<?> clazz, String name, Object bean, String beanName) {
        if (isEmpty(name)) {
            name = clazz.getSimpleName();
        }
        if (bean == null) {
            // no bean so then create an instance from its type
            bean = camelContext.getInjector().newInstance(clazz);
        }
        camelContext.getRegistry().bind(name, bean);
    }

    private void bindToRegistry(Field field, String name, Object bean, String beanName) {
        if (isEmpty(name)) {
            name = field.getName();
        }
        Object value = ReflectionHelper.getField(field, bean);
        if (value != null) {
            camelContext.getRegistry().bind(name, value);
        }
    }

    private void bindToRegistry(Method method, String name, Object bean, String beanName) {
        if (isEmpty(name)) {
            name = method.getName();
        }
        Class<?> returnType = method.getReturnType();
        if (returnType == null || returnType == Void.TYPE) {
            throw new IllegalArgumentException("@BindToRegistry on class: " + method.getDeclaringClass()
                + " method: " + method.getName() + " with void return type is not allowed");
        }

        Object value;
        Object[] parameters = bindToRegistryParameterMapping(method);
        if (parameters != null) {
            value = invokeMethod(method, bean, parameters);
        } else {
            value = invokeMethod(method, bean);
        }
        if (value != null) {
            camelContext.getRegistry().bind(name, value);
        }
    }

    private Object[] bindToRegistryParameterMapping(Method method) {
        if (method.getParameterCount() == 0) {
            return null;
        }

        // map each parameter if possible
        Object[] parameters = new Object[method.getParameterCount()];
        for (int i = 0; i < method.getParameterCount(); i++) {
            Class<?> type = method.getParameterTypes()[i];
            if (type.isAssignableFrom(CamelContext.class)) {
                parameters[i] = camelContext;
            } else if (type.isAssignableFrom(Registry.class)) {
                parameters[i] = camelContext.getRegistry();
            } else if (type.isAssignableFrom(TypeConverter.class)) {
                parameters[i] = camelContext.getTypeConverter();
            } else {
                // we also support @BeanInject and @PropertyInject annotations
                Annotation[] anns = method.getParameterAnnotations()[i];
                if (anns.length == 1) {
                    // we dont assume there are multiple annotations on the same parameter so grab first
                    Annotation ann = anns[0];
                    if (ann.annotationType() == PropertyInject.class) {
                        PropertyInject pi = (PropertyInject) ann;
                        // build key with default value included as this is supported during resolving
                        String key = pi.value();
                        if (!isEmpty(pi.defaultValue())) {
                            key = key + ":" + pi.defaultValue();
                        }
                        // need to force property lookup by having key enclosed in tokens
                        key = camelContext.getPropertiesComponent().getPrefixToken() + key + camelContext.getPropertiesComponent().getSuffixToken();
                        try {
                            Object value = camelContext.resolvePropertyPlaceholders(key);
                            parameters[i] = camelContext.getTypeConverter().convertTo(type, value);
                        } catch (Exception e) {
                            throw RuntimeCamelException.wrapRuntimeCamelException(e);
                        }
                    } else if (ann.annotationType() == BeanInject.class) {
                        BeanInject bi = (BeanInject) ann;
                        String key = bi.value();
                        Object value;
                        if (isEmpty(key)) {
                            // empty key so lookup anonymously by type
                            Set<?> instances = camelContext.getRegistry().findByType(type);
                            if (instances.size() == 0) {
                                throw new NoSuchBeanException(null, key);
                            } else if (instances.size() == 1) {
                                parameters[i] = instances.iterator().next();
                            } else {
                                // there are multiple instances of the same type, so barf
                                throw new IllegalArgumentException("Multiple beans of the same type: " + type
                                    + " exists in the Camel registry. Specify the bean name on @BeanInject to bind to a single bean, at the method: " + method);
                            }
                        } else {
                            value = camelContext.getRegistry().lookupByName(key);
                            if (value == null) {
                                throw new NoSuchBeanException(key);
                            }
                            parameters[i] = camelContext.getTypeConverter().convertTo(type, value);
                        }
                    }
                }
            }

            // each parameter must be mapped
            if (parameters[i] == null) {
                int pos = i + 1;
                throw new IllegalArgumentException("@BindToProperty cannot bind parameter #" + pos + " on method: " + method);
            }
        }

        return parameters;
    }

}
