/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.file.remote;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.apache.camel.Exchange;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.component.file.GenericFile;
import org.apache.camel.component.file.GenericFileEndpoint;
import org.apache.camel.component.file.GenericFileExist;
import org.apache.camel.component.file.GenericFileOperationFailedException;
import org.apache.camel.component.file.remote.DefaultFtpClientActivityListener;
import org.apache.camel.component.file.remote.FtpClientActivityListener;
import org.apache.camel.component.file.remote.FtpConfiguration;
import org.apache.camel.component.file.remote.FtpEndpoint;
import org.apache.camel.component.file.remote.FtpUtils;
import org.apache.camel.component.file.remote.RemoteFileConfiguration;
import org.apache.camel.component.file.remote.RemoteFileOperations;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.apache.camel.util.StopWatch;
import org.apache.camel.util.TimeUtils;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPClientConfig;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;
import org.apache.commons.net.io.CopyStreamListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FtpOperations
implements RemoteFileOperations<FTPFile> {
    protected final Logger log = LoggerFactory.getLogger(this.getClass());
    protected final FTPClient client;
    protected final FTPClientConfig clientConfig;
    protected FtpEndpoint<FTPFile> endpoint;
    protected FtpClientActivityListener clientActivityListener;

    public FtpOperations(FTPClient client, FTPClientConfig clientConfig) {
        this.client = client;
        this.clientConfig = clientConfig;
    }

    public void setEndpoint(GenericFileEndpoint<FTPFile> endpoint) {
        this.endpoint = (FtpEndpoint)endpoint;
        this.clientActivityListener = new DefaultFtpClientActivityListener(this.endpoint, this.endpoint.getConfiguration().remoteServerInformation());
    }

    public FtpClientActivityListener getClientActivityListener() {
        return this.clientActivityListener;
    }

    public void setClientActivityListener(FtpClientActivityListener clientActivityListener) {
        this.clientActivityListener = clientActivityListener;
    }

    @Override
    public boolean connect(RemoteFileConfiguration configuration) throws GenericFileOperationFailedException {
        this.client.setCopyStreamListener((CopyStreamListener)this.clientActivityListener);
        try {
            return this.doConnect(configuration);
        }
        catch (GenericFileOperationFailedException e) {
            this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
            throw e;
        }
    }

    protected boolean doConnect(RemoteFileConfiguration configuration) throws GenericFileOperationFailedException {
        FtpEndpoint<FTPFile> ftpEndpoint;
        this.log.trace("Connecting using FTPClient: {}", (Object)this.client);
        String host = configuration.getHost();
        int port = configuration.getPort();
        String username = configuration.getUsername();
        String account = ((FtpConfiguration)configuration).getAccount();
        if (this.clientConfig != null) {
            this.log.trace("Configuring FTPClient with config: {}", (Object)this.clientConfig);
            this.client.configure(this.clientConfig);
        }
        if (this.log.isTraceEnabled()) {
            this.log.trace("Connecting to {} using connection timeout: {}", (Object)configuration.remoteServerInformation(), (Object)this.client.getConnectTimeout());
        }
        boolean connected = false;
        int attempt = 0;
        while (!connected) {
            try {
                if (this.log.isTraceEnabled() && attempt > 0) {
                    this.log.trace("Reconnect attempt #{} connecting to {}", (Object)attempt, (Object)configuration.remoteServerInformation());
                }
                this.clientActivityListener.onConnecting(host);
                this.client.connect(host, port);
                int reply = this.client.getReplyCode();
                if (FTPReply.isPositiveCompletion((int)reply)) {
                    connected = true;
                    continue;
                }
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), "Server refused connection");
            }
            catch (Exception e) {
                if (this.client.isConnected()) {
                    this.log.trace("Disconnecting due to exception during connect");
                    try {
                        this.client.disconnect();
                    }
                    catch (IOException ignore) {
                        this.log.trace("Ignore exception during disconnect: {}", (Object)ignore.getMessage());
                    }
                }
                if (Thread.currentThread().isInterrupted()) {
                    throw new GenericFileOperationFailedException("Interrupted during connecting", (Throwable)new InterruptedException("Interrupted during connecting"));
                }
                GenericFileOperationFailedException failed = e instanceof GenericFileOperationFailedException ? (GenericFileOperationFailedException)((Object)e) : new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
                this.log.trace("Cannot connect due: {}", (Object)failed.getMessage());
                if (++attempt > this.endpoint.getMaximumReconnectAttempts()) {
                    throw failed;
                }
                if (this.endpoint.getReconnectDelay() <= 0L) continue;
                try {
                    Thread.sleep(this.endpoint.getReconnectDelay());
                }
                catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new GenericFileOperationFailedException("Interrupted during sleeping", (Throwable)ie);
                }
            }
        }
        this.clientActivityListener.onConnected(host);
        if (configuration.isPassiveMode()) {
            this.log.trace("Using passive mode connections");
            this.client.enterLocalPassiveMode();
        }
        if (this.endpoint instanceof FtpEndpoint && (ftpEndpoint = this.endpoint).getSoTimeout() > 0) {
            this.log.trace("Using SoTimeout={}", (Object)ftpEndpoint.getSoTimeout());
            try {
                this.client.setSoTimeout(ftpEndpoint.getSoTimeout());
            }
            catch (IOException e) {
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
            }
        }
        try {
            boolean login;
            this.clientActivityListener.onLogin(host);
            if (username != null) {
                if (account != null) {
                    this.log.trace("Attempting to login user: {} using password: ******** and account: {}", new Object[]{username, account});
                    login = this.client.login(username, configuration.getPassword(), account);
                } else {
                    this.log.trace("Attempting to login user: {} using password: ********", (Object)username);
                    login = this.client.login(username, configuration.getPassword());
                }
            } else if (account != null) {
                this.log.trace("Attempting to login anonymous using account: {}", (Object)account);
                login = this.client.login("anonymous", "", account);
            } else {
                this.log.trace("Attempting to login anonymous");
                login = this.client.login("anonymous", "");
            }
            this.log.trace("User {} logged in: {}", (Object)(username != null ? username : "anonymous"), (Object)login);
            if (!login) {
                String replyString = this.client.getReplyString();
                int replyCode = this.client.getReplyCode();
                this.clientActivityListener.onLoginFailed(replyCode, replyString);
                this.client.disconnect();
                throw new GenericFileOperationFailedException(replyCode, replyString);
            }
            this.clientActivityListener.onLoginComplete(host);
            this.client.setFileType(configuration.isBinary() ? 2 : 0);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        if (this.endpoint.getConfiguration().getSiteCommand() != null) {
            Iterator it = ObjectHelper.createIterator((Object)this.endpoint.getConfiguration().getSiteCommand(), (String)"\n");
            while (it.hasNext()) {
                boolean result;
                Object next = it.next();
                String command = (String)this.endpoint.getCamelContext().getTypeConverter().convertTo(String.class, next);
                this.log.trace("Site command to send: {}", (Object)command);
                if (command == null || (result = this.sendSiteCommand(command))) continue;
                throw new GenericFileOperationFailedException("Site command: " + command + " returned false");
            }
        }
        return true;
    }

    @Override
    public boolean isConnected() throws GenericFileOperationFailedException {
        return this.client.isConnected();
    }

    @Override
    public void disconnect() throws GenericFileOperationFailedException {
        try {
            this.doDisconnect();
        }
        catch (GenericFileOperationFailedException e) {
            this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
            throw e;
        }
    }

    @Override
    public void forceDisconnect() throws GenericFileOperationFailedException {
        this.doDisconnect();
    }

    protected void doDisconnect() throws GenericFileOperationFailedException {
        this.clientActivityListener.onDisconnecting(this.endpoint.getConfiguration().remoteServerInformation());
        try {
            this.log.trace("Client logout");
            this.client.logout();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        finally {
            try {
                this.log.trace("Client disconnect");
                this.client.disconnect();
            }
            catch (IOException e) {
                throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
            }
        }
        this.clientActivityListener.onDisconnected(this.endpoint.getConfiguration().remoteServerInformation());
    }

    public boolean deleteFile(String name) throws GenericFileOperationFailedException {
        boolean result;
        this.log.debug("Deleting file: {}", (Object)name);
        String target = name;
        String currentDir = null;
        try {
            if (this.endpoint.getConfiguration().isStepwise()) {
                currentDir = this.getCurrentDirectory();
                target = FileUtil.stripPath((String)name);
                try {
                    this.changeCurrentDirectory(FileUtil.onlyPath((String)name));
                }
                catch (GenericFileOperationFailedException e) {
                    this.changeCurrentDirectory(currentDir);
                    throw e;
                }
            }
            this.log.trace("Client deleteFile: {}", (Object)target);
            result = this.client.deleteFile(target);
            if (currentDir != null) {
                this.changeCurrentDirectory(currentDir);
            }
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        return result;
    }

    public boolean renameFile(String from, String to) throws GenericFileOperationFailedException {
        this.log.debug("Renaming file: {} to: {}", (Object)from, (Object)to);
        try {
            return this.client.rename(from, to);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean buildDirectory(String directory, boolean absolute) throws GenericFileOperationFailedException {
        boolean bl;
        block8: {
            directory = this.endpoint.getConfiguration().normalizePath(directory);
            this.log.trace("buildDirectory({})", (Object)directory);
            String originalDirectory = this.client.printWorkingDirectory();
            try {
                boolean success = this.client.changeWorkingDirectory(directory);
                if (!success) {
                    this.log.trace("Trying to build remote directory: {}", (Object)directory);
                    success = this.client.makeDirectory(directory);
                    if (!success) {
                        success = this.buildDirectoryChunks(directory);
                    }
                }
                bl = success;
                if (originalDirectory == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (originalDirectory != null) {
                        this.changeCurrentDirectory(originalDirectory);
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
                }
            }
            this.changeCurrentDirectory(originalDirectory);
        }
        return bl;
    }

    public boolean retrieveFile(String name, Exchange exchange, long size) throws GenericFileOperationFailedException {
        boolean answer;
        this.clientActivityListener.setDownload(true);
        this.clientActivityListener.setRemoteFileName(name);
        this.clientActivityListener.setRemoteFileSize(size);
        this.clientActivityListener.onBeginDownloading(this.endpoint.getConfiguration().remoteServerInformation(), name);
        try {
            this.log.trace("retrieveFile({})", (Object)name);
            answer = ObjectHelper.isNotEmpty((Object)this.endpoint.getLocalWorkDirectory()) ? this.retrieveFileToFileInLocalWorkDirectory(name, exchange, this.endpoint.isResumeDownload()) : this.retrieveFileToStreamInBody(name, exchange);
        }
        catch (GenericFileOperationFailedException e) {
            this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
            throw e;
        }
        if (answer) {
            this.clientActivityListener.onDownloadComplete(this.endpoint.getConfiguration().remoteServerInformation(), name);
        }
        return answer;
    }

    public void releaseRetrievedFileResources(Exchange exchange) throws GenericFileOperationFailedException {
        InputStream is = (InputStream)exchange.getIn().getHeader("CamelRemoteFileInputStream", InputStream.class);
        if (is != null) {
            try {
                IOHelper.close((Closeable)is);
                this.client.completePendingCommand();
            }
            catch (IOException e) {
                throw new GenericFileOperationFailedException(e.getMessage(), (Throwable)e);
            }
        }
    }

    private boolean retrieveFileToStreamInBody(String name, Exchange exchange) throws GenericFileOperationFailedException {
        boolean result;
        try {
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            ObjectHelper.notNull((Object)target, (String)"Exchange should have the CamelFileExchangeFile set");
            String remoteName = name;
            String currentDir = null;
            if (this.endpoint.getConfiguration().isStepwise()) {
                currentDir = this.getCurrentDirectory();
                String path = FileUtil.onlyPath((String)name);
                if (path != null) {
                    this.changeCurrentDirectory(path);
                }
                remoteName = FileUtil.stripPath((String)name);
            }
            this.log.trace("Client retrieveFile: {}", (Object)remoteName);
            if (this.endpoint.getConfiguration().isStreamDownload()) {
                InputStream is = this.client.retrieveFileStream(remoteName);
                target.setBody((Object)is);
                exchange.getIn().setHeader("CamelRemoteFileInputStream", (Object)is);
                result = true;
            } else {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                result = this.client.retrieveFile(remoteName, (OutputStream)bos);
                IOHelper.close((Closeable)bos);
                target.setBody((Object)bos.toByteArray());
            }
            exchange.getIn().setHeader("CamelFtpReplyCode", (Object)this.client.getReplyCode());
            exchange.getIn().setHeader("CamelFtpReplyString", (Object)this.client.getReplyString());
            if (this.endpoint.getConfiguration().isStepwise()) {
                this.changeCurrentDirectory(currentDir);
            }
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        return result;
    }

    private boolean retrieveFileToFileInLocalWorkDirectory(String name, Exchange exchange, boolean resumeDownload) throws GenericFileOperationFailedException {
        boolean result;
        FileOutputStream os;
        File temp;
        File local = new File(FileUtil.normalizePath((String)this.endpoint.getLocalWorkDirectory()));
        long existingSize = -1L;
        try {
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            ObjectHelper.notNull((Object)target, (String)"Exchange should have the CamelFileExchangeFile set");
            String relativeName = target.getRelativeFilePath();
            temp = new File(local, relativeName + ".inprogress");
            local = new File(local, relativeName);
            local.mkdirs();
            if (local.exists() && !FileUtil.deleteFile((File)local)) {
                throw new GenericFileOperationFailedException("Cannot delete existing local work file: " + local);
            }
            boolean exists = temp.exists();
            if (exists) {
                existingSize = temp.length();
            }
            if (!resumeDownload) {
                if (exists && !FileUtil.deleteFile((File)temp)) {
                    throw new GenericFileOperationFailedException("Cannot delete existing local work file: " + temp);
                }
                if (!temp.createNewFile()) {
                    throw new GenericFileOperationFailedException("Cannot create new local work file: " + temp);
                }
            }
            boolean append = resumeDownload && existingSize > 0L;
            os = new FileOutputStream(temp, append);
            exchange.getIn().setHeader("CamelFileLocalWorkPath", (Object)local.getPath());
        }
        catch (Exception e) {
            throw new GenericFileOperationFailedException("Cannot create new local work file: " + local);
        }
        try {
            GenericFile target = (GenericFile)exchange.getProperty("CamelFileExchangeFile");
            target.setBody((Object)local);
            String remoteName = name;
            String currentDir = null;
            if (this.endpoint.getConfiguration().isStepwise()) {
                currentDir = this.getCurrentDirectory();
                String path = FileUtil.onlyPath((String)name);
                if (path != null) {
                    this.changeCurrentDirectory(path);
                }
                remoteName = FileUtil.stripPath((String)name);
            }
            if (resumeDownload && existingSize > 0L) {
                this.clientActivityListener.onResumeDownloading(this.endpoint.getConfiguration().remoteServerInformation(), name, existingSize);
                this.log.trace("Client restartOffset: {}", (Object)existingSize);
                this.log.debug("Resuming download of file: {} at position: {}", (Object)remoteName, (Object)existingSize);
                this.client.setRestartOffset(existingSize);
            }
            this.log.trace("Client retrieveFile: {}", (Object)remoteName);
            result = this.client.retrieveFile(remoteName, (OutputStream)os);
            exchange.getIn().setHeader("CamelFtpReplyCode", (Object)this.client.getReplyCode());
            exchange.getIn().setHeader("CamelFtpReplyString", (Object)this.client.getReplyString());
            if (this.endpoint.getConfiguration().isStepwise()) {
                this.changeCurrentDirectory(currentDir);
            }
        }
        catch (IOException e) {
            this.log.trace("Error occurred during retrieving file: {} to local directory.", (Object)name);
            if (!resumeDownload) {
                this.log.trace("Deleting local work file: {}", (Object)name, (Object)temp);
                IOHelper.close((Closeable)os, (String)("retrieve: " + name), (Logger)this.log);
                boolean deleted = FileUtil.deleteFile((File)temp);
                if (!deleted) {
                    this.log.warn("Error occurred during retrieving file: " + name + " to local directory. Cannot delete local work file: " + temp);
                }
            }
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        finally {
            IOHelper.close((Closeable)os, (String)("retrieve: " + name), (Logger)this.log);
        }
        this.log.debug("Retrieve file to local work file result: {}", (Object)result);
        if (result) {
            this.log.trace("Renaming local in progress file from: {} to: {}", (Object)temp, (Object)local);
            try {
                if (!FileUtil.renameFile((File)temp, (File)local, (boolean)false)) {
                    throw new GenericFileOperationFailedException("Cannot rename local work file from: " + temp + " to: " + local);
                }
            }
            catch (IOException e) {
                throw new GenericFileOperationFailedException("Cannot rename local work file from: " + temp + " to: " + local, (Throwable)e);
            }
        }
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean storeFile(String name, Exchange exchange, long size) throws GenericFileOperationFailedException {
        boolean answer;
        block7: {
            name = this.endpoint.getConfiguration().normalizePath(name);
            this.clientActivityListener.setDownload(false);
            this.clientActivityListener.setRemoteFileName(name);
            this.clientActivityListener.setRemoteFileSize(size);
            this.clientActivityListener.onBeginUploading(this.endpoint.getConfiguration().remoteServerInformation(), name);
            this.log.trace("storeFile({})", (Object)name);
            String currentDir = null;
            String path = FileUtil.onlyPath((String)name);
            String targetName = name;
            try {
                if (path != null && this.endpoint.getConfiguration().isStepwise()) {
                    currentDir = this.getCurrentDirectory();
                    this.changeCurrentDirectory(path);
                    targetName = FileUtil.stripPath((String)name);
                }
                answer = this.doStoreFile(name, targetName, exchange);
                if (currentDir == null) break block7;
            }
            catch (GenericFileOperationFailedException e) {
                try {
                    this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
                    throw e;
                }
                catch (Throwable throwable) {
                    if (currentDir != null) {
                        this.changeCurrentDirectory(currentDir);
                    }
                    throw throwable;
                }
            }
            this.changeCurrentDirectory(currentDir);
        }
        if (answer) {
            this.clientActivityListener.onUploadComplete(this.endpoint.getConfiguration().remoteServerInformation(), name);
        }
        return answer;
    }

    private boolean doStoreFile(String name, String targetName, Exchange exchange) throws GenericFileOperationFailedException {
        this.log.trace("doStoreFile({})", (Object)targetName);
        if (this.endpoint.getFileExist() == GenericFileExist.Ignore || this.endpoint.getFileExist() == GenericFileExist.Fail || this.endpoint.getFileExist() == GenericFileExist.Move) {
            boolean existFile = this.existsFile(targetName);
            if (existFile && this.endpoint.getFileExist() == GenericFileExist.Ignore) {
                this.log.trace("An existing file already exists: {}. Ignore and do not override it.", (Object)name);
                return true;
            }
            if (existFile && this.endpoint.getFileExist() == GenericFileExist.Fail) {
                throw new GenericFileOperationFailedException("File already exist: " + name + ". Cannot write new file.");
            }
            if (existFile && this.endpoint.getFileExist() == GenericFileExist.Move) {
                this.doMoveExistingFile(name, targetName);
            }
        }
        InputStream is = null;
        if (exchange.getIn().getBody() == null) {
            if (this.endpoint.isAllowNullBody()) {
                this.log.trace("Writing empty file.");
                is = new ByteArrayInputStream(new byte[0]);
            } else {
                throw new GenericFileOperationFailedException("Cannot write null body to file: " + name);
            }
        }
        try {
            boolean answer;
            if (is == null) {
                String charset = this.endpoint.getCharset();
                if (charset != null) {
                    is = new ByteArrayInputStream(((String)exchange.getIn().getMandatoryBody(String.class)).getBytes(charset));
                    this.log.trace("Using InputStream {} with charset {}.", (Object)is, (Object)charset);
                } else {
                    is = (InputStream)exchange.getIn().getMandatoryBody(InputStream.class);
                }
            }
            StopWatch watch = new StopWatch();
            this.log.debug("About to store file: {} using stream: {}", (Object)targetName, is);
            if (this.endpoint.getFileExist() == GenericFileExist.Append) {
                this.log.trace("Client appendFile: {}", (Object)targetName);
                answer = this.client.appendFile(targetName, is);
            } else {
                this.log.trace("Client storeFile: {}", (Object)targetName);
                answer = this.client.storeFile(targetName, is);
            }
            if (this.log.isDebugEnabled()) {
                long time = watch.taken();
                this.log.debug("Took {} ({} millis) to store file: {} and FTP client returned: {}", new Object[]{TimeUtils.printDuration((double)time), time, targetName, answer});
            }
            exchange.getIn().setHeader("CamelFtpReplyCode", (Object)this.client.getReplyCode());
            exchange.getIn().setHeader("CamelFtpReplyString", (Object)this.client.getReplyString());
            String chmod = this.endpoint.getConfiguration().getChmod();
            if (ObjectHelper.isNotEmpty((Object)chmod)) {
                this.log.debug("Setting chmod: {} on file: {}", (Object)chmod, (Object)targetName);
                String command = "chmod " + chmod + " " + targetName;
                this.log.trace("Client sendSiteCommand: {}", (Object)command);
                boolean success = this.client.sendSiteCommand(command);
                this.log.trace("Client sendSiteCommand successful: {}", (Object)success);
            }
            boolean bl = answer;
            return bl;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        catch (InvalidPayloadException e) {
            throw new GenericFileOperationFailedException("Cannot store file: " + name, (Throwable)e);
        }
        finally {
            IOHelper.close((Closeable)is, (String)("store: " + name), (Logger)this.log);
        }
    }

    private void doMoveExistingFile(String name, String targetName) throws GenericFileOperationFailedException {
        String to;
        block8: {
            Exchange dummy = this.endpoint.createExchange();
            Object parent = null;
            String onlyName = FileUtil.stripPath((String)targetName);
            dummy.getIn().setHeader("CamelFileName", (Object)targetName);
            dummy.getIn().setHeader("CamelFileNameOnly", (Object)onlyName);
            dummy.getIn().setHeader("CamelFileParent", parent);
            to = (String)this.endpoint.getMoveExisting().evaluate(dummy, String.class);
            to = FileUtil.stripLeadingSeparator((String)to);
            to = this.endpoint.getConfiguration().normalizePath(to);
            if (ObjectHelper.isEmpty((Object)to)) {
                throw new GenericFileOperationFailedException("moveExisting evaluated as empty String, cannot move existing file: " + name);
            }
            String dir = FileUtil.onlyPath((String)to);
            if (dir != null) {
                this.buildDirectory(dir, false);
            }
            if (this.existsFile(to)) {
                if (this.endpoint.isEagerDeleteTargetFile()) {
                    this.log.trace("Deleting existing file: {}", (Object)to);
                    try {
                        boolean result = this.client.deleteFile(to);
                        if (!result) {
                            throw new GenericFileOperationFailedException("Cannot delete file: " + to);
                        }
                        break block8;
                    }
                    catch (IOException e) {
                        throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), "Cannot delete file: " + to, (Throwable)e);
                    }
                }
                throw new GenericFileOperationFailedException("Cannot moved existing file from: " + name + " to: " + to + " as there already exists a file: " + to);
            }
        }
        this.log.trace("Moving existing file: {} to: {}", (Object)name, (Object)to);
        if (!this.renameFile(targetName, to)) {
            throw new GenericFileOperationFailedException("Cannot rename file from: " + name + " to: " + to);
        }
    }

    public boolean existsFile(String name) throws GenericFileOperationFailedException {
        this.log.trace("existsFile({})", (Object)name);
        if (this.endpoint.isFastExistsCheck()) {
            return this.fastExistsFile(name);
        }
        String directory = FileUtil.onlyPath((String)name);
        String onlyName = FileUtil.stripPath((String)name);
        try {
            String[] names = directory != null ? this.client.listNames(directory) : this.client.listNames();
            if (names == null) {
                return false;
            }
            for (String existing : names) {
                this.log.trace("Existing file: {}, target file: {}", (Object)existing, (Object)name);
                existing = FileUtil.stripPath((String)existing);
                if (existing == null || !existing.equals(onlyName)) continue;
                return true;
            }
            return false;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    protected boolean fastExistsFile(String name) throws GenericFileOperationFailedException {
        this.log.trace("fastExistsFile({})", (Object)name);
        try {
            String[] names = this.client.listNames(name);
            if (names == null) {
                return false;
            }
            return names.length >= 1;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public String getCurrentDirectory() throws GenericFileOperationFailedException {
        this.log.trace("getCurrentDirectory()");
        try {
            String answer = this.client.printWorkingDirectory();
            this.log.trace("Current dir: {}", (Object)answer);
            return answer;
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public void changeCurrentDirectory(String path) throws GenericFileOperationFailedException {
        String[] dirs;
        this.log.trace("changeCurrentDirectory({})", (Object)path);
        if (ObjectHelper.isEmpty((Object)path)) {
            return;
        }
        path = FtpUtils.compactPath(path);
        if (!this.endpoint.getConfiguration().isStepwise()) {
            this.doChangeDirectory(path);
            return;
        }
        if (FileUtil.hasLeadingSeparator((String)path)) {
            this.doChangeDirectory(path.substring(0, 1));
            path = path.substring(1);
        }
        if ((dirs = path.split("/|\\\\")) == null || dirs.length == 0) {
            this.doChangeDirectory(path);
            return;
        }
        for (String dir : dirs) {
            this.doChangeDirectory(dir);
        }
    }

    private void doChangeDirectory(String path) {
        boolean success;
        if (path == null || ".".equals(path) || ObjectHelper.isEmpty((Object)path)) {
            return;
        }
        this.log.trace("Changing directory: {}", (Object)path);
        try {
            if ("..".equals(path)) {
                this.changeToParentDirectory();
                success = true;
            } else {
                success = this.client.changeWorkingDirectory(path);
            }
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
        if (!success) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), "Cannot change directory to: " + path);
        }
    }

    public void changeToParentDirectory() throws GenericFileOperationFailedException {
        try {
            this.client.changeToParentDirectory();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public List<FTPFile> listFiles() throws GenericFileOperationFailedException {
        this.log.trace("listFiles()");
        this.clientActivityListener.onScanningForFiles(this.endpoint.remoteServerInformation(), null);
        try {
            ArrayList<FTPFile> list = new ArrayList<FTPFile>();
            FTPFile[] files = this.client.listFiles();
            if (files != null) {
                list.addAll(Arrays.asList(files));
            }
            return list;
        }
        catch (IOException e) {
            this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    public List<FTPFile> listFiles(String path) throws GenericFileOperationFailedException {
        this.log.trace("listFiles({})", (Object)path);
        this.clientActivityListener.onScanningForFiles(this.endpoint.remoteServerInformation(), path);
        if (ObjectHelper.isEmpty((Object)path)) {
            path = ".";
        }
        try {
            ArrayList<FTPFile> list = new ArrayList<FTPFile>();
            FTPFile[] files = this.client.listFiles(path);
            if (files != null) {
                list.addAll(Arrays.asList(files));
            }
            return list;
        }
        catch (IOException e) {
            this.clientActivityListener.onGeneralError(this.endpoint.getConfiguration().remoteServerInformation(), e.getMessage());
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    @Override
    public boolean sendNoop() throws GenericFileOperationFailedException {
        this.log.trace("sendNoOp");
        try {
            return this.client.sendNoOp();
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    @Override
    public boolean sendSiteCommand(String command) throws GenericFileOperationFailedException {
        this.log.trace("sendSiteCommand({})", (Object)command);
        try {
            return this.client.sendSiteCommand(command);
        }
        catch (IOException e) {
            throw new GenericFileOperationFailedException(this.client.getReplyCode(), this.client.getReplyString(), e.getMessage(), (Throwable)e);
        }
    }

    protected FTPClient getFtpClient() {
        return this.client;
    }

    private boolean buildDirectoryChunks(String dirName) throws IOException {
        StringBuilder sb = new StringBuilder(dirName.length());
        String[] dirs = dirName.split("/|\\\\");
        boolean success = false;
        for (String dir : dirs) {
            sb.append(dir).append('/');
            String directory = this.endpoint.getConfiguration().normalizePath(sb.toString());
            if (directory.equals("/") || directory.equals("\\")) continue;
            this.log.trace("Trying to build remote directory by chunk: {}", (Object)directory);
            directory = FileUtil.stripTrailingSeparator((String)directory);
            success = this.client.makeDirectory(directory);
        }
        return success;
    }
}

