/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.mllp.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.SocketTimeoutException;
import java.nio.charset.Charset;
import java.util.Arrays;
import org.apache.camel.component.mllp.MllpComponent;
import org.apache.camel.component.mllp.MllpEndpoint;
import org.apache.camel.component.mllp.MllpProtocolConstants;
import org.apache.camel.component.mllp.MllpSocketException;
import org.apache.camel.component.mllp.internal.Hl7Util;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MllpSocketBuffer {
    static final int MIN_BUFFER_SIZE = 2048;
    static final int MAX_BUFFER_SIZE = 0x40000000;
    final Logger log = LoggerFactory.getLogger(this.getClass());
    final MllpEndpoint endpoint;
    byte[] buffer;
    int availableByteCount;
    int startOfBlockIndex = -1;
    int endOfBlockIndex = -1;

    public MllpSocketBuffer(MllpEndpoint endpoint) {
        if (endpoint == null) {
            throw new IllegalArgumentException("MllpEndpoint cannot be null");
        }
        this.endpoint = endpoint;
        this.buffer = new byte[2048];
    }

    public boolean isEndOfDataRequired() {
        return this.endpoint.getConfiguration().isRequireEndOfData();
    }

    public boolean isEmpty() {
        return this.size() <= 0;
    }

    public synchronized void write(int b) {
        this.ensureCapacity(1);
        this.buffer[this.availableByteCount] = (byte)b;
        this.updateIndexes(b, 0);
        ++this.availableByteCount;
    }

    public void write(byte[] b) {
        if (b != null && b.length > 0) {
            this.write(b, 0, b.length);
        }
    }

    public synchronized void write(byte[] sourceBytes, int offset, int writeCount) {
        if (sourceBytes != null && sourceBytes.length > 0) {
            if (offset < 0) {
                throw new IndexOutOfBoundsException(String.format("write(byte[%d], offset[%d], writeCount[%d]) - offset is less than zero", sourceBytes.length, offset, writeCount));
            }
            if (offset > sourceBytes.length) {
                throw new IndexOutOfBoundsException(String.format("write(byte[%d], offset[%d], writeCount[%d]) - offset is greater than write count", sourceBytes.length, offset, writeCount));
            }
            if (writeCount < 0) {
                throw new IndexOutOfBoundsException(String.format("write(byte[%d], offset[%d], writeCount[%d]) - write count is less than zero", sourceBytes.length, offset, writeCount));
            }
            if (writeCount > sourceBytes.length) {
                throw new IndexOutOfBoundsException(String.format("write(byte[%d], offset[%d], writeCount[%d]) - write count is greater than length of the source byte[]", sourceBytes.length, offset, writeCount));
            }
            if (offset + writeCount - sourceBytes.length > 0) {
                throw new IndexOutOfBoundsException(String.format("write(byte[%d], offset[%d], writeCount[%d]) - offset plus write count <%d> is greater than length of the source byte[]", sourceBytes.length, offset, writeCount, offset + writeCount));
            }
            this.ensureCapacity(writeCount);
            System.arraycopy(sourceBytes, offset, this.buffer, this.availableByteCount, writeCount);
            for (int i = offset; i < writeCount && (this.startOfBlockIndex < 0 || this.endOfBlockIndex < 0); ++i) {
                this.updateIndexes(sourceBytes[i], i);
            }
            this.availableByteCount += writeCount;
        }
    }

    public synchronized void openMllpEnvelope() {
        this.reset();
        this.write(11);
    }

    public synchronized void closeMllpEnvelope() {
        this.write(MllpProtocolConstants.PAYLOAD_TERMINATOR);
    }

    public synchronized void setEnvelopedMessage(byte[] hl7Payload) {
        this.setEnvelopedMessage(hl7Payload, 0, hl7Payload != null ? hl7Payload.length : 0);
    }

    public synchronized void setEnvelopedMessage(byte[] hl7Payload, int offset, int length) {
        this.reset();
        if (hl7Payload != null && hl7Payload.length > 0) {
            if (hl7Payload[0] != 11) {
                this.openMllpEnvelope();
            }
            this.write(hl7Payload, offset, length);
            if (!this.hasCompleteEnvelope()) {
                this.closeMllpEnvelope();
            }
        } else {
            this.openMllpEnvelope();
            this.closeMllpEnvelope();
        }
    }

    public synchronized void reset() {
        if (this.availableByteCount > 0) {
            Arrays.fill(this.buffer, (byte)0);
        }
        this.availableByteCount = 0;
        this.startOfBlockIndex = -1;
        this.endOfBlockIndex = -1;
    }

    public synchronized void readFrom(Socket socket) throws MllpSocketException, SocketTimeoutException {
        this.readFrom(socket, this.endpoint.getConfiguration().getReceiveTimeout(), this.endpoint.getConfiguration().getReadTimeout());
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public synchronized void readFrom(Socket socket, int receiveTimeout, int readTimeout) throws MllpSocketException, SocketTimeoutException {
        block10: {
            if (socket != null && socket.isConnected() && !socket.isClosed()) {
                this.log.trace("readFrom({}, {}, {}) - entering", new Object[]{socket, receiveTimeout, readTimeout});
                this.ensureCapacity(2048);
                try {
                    InputStream socketInputStream = socket.getInputStream();
                    socket.setSoTimeout(receiveTimeout);
                    this.readSocketInputStream(socketInputStream, socket);
                    if (!this.hasCompleteEnvelope()) {
                        socket.setSoTimeout(readTimeout);
                        while (!this.hasCompleteEnvelope()) {
                            this.ensureCapacity(Math.max(2048, socketInputStream.available()));
                            this.readSocketInputStream(socketInputStream, socket);
                        }
                    }
                    if (this.size() <= 0 || this.hasCompleteEnvelope() || this.hasEndOfData() || !this.hasEndOfBlock() || this.endOfBlockIndex >= this.size() - 1) break block10;
                }
                catch (SocketTimeoutException timeoutEx) {
                    try {
                        throw timeoutEx;
                        catch (IOException ioEx) {
                            String exceptionMessage = String.format("readFrom(%s, %d, %d) - IOException encountered", socket, receiveTimeout, readTimeout);
                            this.resetSocket(socket, exceptionMessage);
                            throw new MllpSocketException(exceptionMessage, ioEx);
                        }
                    }
                    catch (Throwable throwable) {
                        if (this.size() > 0 && !this.hasCompleteEnvelope() && !this.hasEndOfData() && this.hasEndOfBlock() && this.endOfBlockIndex < this.size() - 1) {
                            this.log.warn("readFrom({}, {}, {}) - exiting with partial payload {}", new Object[]{socket, receiveTimeout, readTimeout, Hl7Util.convertToPrintFriendlyString(this.buffer, 0, this.size() - 1)});
                        }
                        throw throwable;
                    }
                }
                this.log.warn("readFrom({}, {}, {}) - exiting with partial payload {}", new Object[]{socket, receiveTimeout, readTimeout, Hl7Util.convertToPrintFriendlyString(this.buffer, 0, this.size() - 1)});
            } else {
                this.log.warn("readFrom({}, {}, {}) - no data read because Socket is invalid", new Object[]{socket, receiveTimeout, readTimeout});
            }
        }
        this.log.trace("readFrom({}, {}, {}) - exiting", new Object[]{socket, receiveTimeout, readTimeout});
    }

    public synchronized void writeTo(Socket socket) throws MllpSocketException {
        if (socket != null && socket.isConnected() && !socket.isClosed()) {
            this.log.trace("writeTo({}) - entering", (Object)socket);
            if (!this.isEmpty()) {
                try {
                    OutputStream socketOutputStream = socket.getOutputStream();
                    if (this.hasStartOfBlock()) {
                        if (this.hasEndOfData()) {
                            socketOutputStream.write(this.buffer, this.startOfBlockIndex, this.endOfBlockIndex - this.startOfBlockIndex + 2);
                        } else if (this.hasEndOfBlock()) {
                            socketOutputStream.write(this.buffer, this.startOfBlockIndex, this.endOfBlockIndex - this.startOfBlockIndex + 1);
                            socketOutputStream.write(13);
                        } else {
                            socketOutputStream.write(this.buffer, this.startOfBlockIndex, this.availableByteCount - this.startOfBlockIndex);
                            socketOutputStream.write(MllpProtocolConstants.PAYLOAD_TERMINATOR);
                        }
                    } else {
                        socketOutputStream.write(11);
                        socketOutputStream.write(this.buffer, 0, this.availableByteCount);
                        socketOutputStream.write(MllpProtocolConstants.PAYLOAD_TERMINATOR);
                    }
                    socketOutputStream.flush();
                }
                catch (IOException ioEx) {
                    String exceptionMessage = String.format("writeTo({}) - IOException encountered", socket);
                    this.resetSocket(socket, exceptionMessage);
                    throw new MllpSocketException(exceptionMessage, ioEx);
                }
            } else {
                this.log.warn("writeTo({}) - no data written because buffer is empty", (Object)socket);
            }
        } else {
            this.log.warn("writeTo({}) - no data written because Socket is invalid", (Object)socket);
        }
        this.log.trace("writeTo({}) - exiting", (Object)socket);
    }

    public synchronized byte[] toByteArray() {
        if (this.availableByteCount > 0) {
            return Arrays.copyOf(this.buffer, this.availableByteCount);
        }
        return null;
    }

    public synchronized byte[] toByteArrayAndReset() {
        byte[] answer = this.toByteArray();
        this.reset();
        return answer;
    }

    public synchronized String toString() {
        return this.toString(MllpComponent.getDefaultCharset());
    }

    public synchronized String toString(Charset charset) {
        if (this.availableByteCount > 0) {
            return new String(this.buffer, 0, this.availableByteCount, charset);
        }
        return "";
    }

    public synchronized String toString(String charsetName) {
        if (this.availableByteCount > 0) {
            try {
                if (Charset.isSupported(charsetName)) {
                    return this.toString(Charset.forName(charsetName));
                }
                this.log.warn("toString(charsetName[{}]) - unsupported character set name - using the MLLP default character set {}", (Object)charsetName, (Object)MllpComponent.getDefaultCharset());
            }
            catch (Exception charsetEx) {
                this.log.warn("toString(charsetName[{}]) - ignoring exception encountered determining character set - using the MLLP default character set {}", new Object[]{charsetName, MllpComponent.getDefaultCharset(), charsetEx});
            }
            return this.toString(MllpComponent.getDefaultCharset());
        }
        return "";
    }

    public synchronized String toStringAndReset() {
        String answer = this.toString();
        this.reset();
        return answer;
    }

    public synchronized String toStringAndReset(String charsetName) {
        String answer = this.toString(charsetName);
        this.reset();
        return answer;
    }

    public synchronized String toPrintFriendlyString() {
        if (this.availableByteCount > 0) {
            return Hl7Util.convertToPrintFriendlyString(this.buffer, 0, this.availableByteCount);
        }
        return "";
    }

    public String toPrintFriendlyStringAndReset() {
        String answer = this.toPrintFriendlyString();
        this.reset();
        return answer;
    }

    public synchronized String toHl7String() {
        return this.toHl7String(MllpComponent.getDefaultCharset());
    }

    public String toHl7StringAndReset() {
        String answer = this.toHl7String();
        this.reset();
        return answer;
    }

    public synchronized String toHl7String(String charsetName) {
        if (charsetName != null && !charsetName.isEmpty()) {
            try {
                if (Charset.isSupported(charsetName)) {
                    return this.toHl7String(Charset.forName(charsetName));
                }
                this.log.warn("toHl7String(charsetName[{}]) - unsupported character set name - using the MLLP default character set {}", (Object)charsetName, (Object)MllpComponent.getDefaultCharset());
            }
            catch (Exception charsetEx) {
                this.log.warn("toHl7String(charsetName[{}]) - ignoring exception encountered determining character set for name - using the MLLP default character set {}", new Object[]{charsetName, MllpComponent.getDefaultCharset(), charsetEx});
            }
        }
        return this.toHl7String(MllpComponent.getDefaultCharset());
    }

    public synchronized String toHl7String(Charset charset) {
        if (this.hasCompleteEnvelope()) {
            int length;
            int offset = this.hasStartOfBlock() ? this.startOfBlockIndex + 1 : 1;
            int n = length = this.hasEndOfBlock() ? this.endOfBlockIndex - offset : this.availableByteCount - this.startOfBlockIndex - 1;
            if (length > 0) {
                return new String(this.buffer, offset, length, charset != null ? charset : MllpComponent.getDefaultCharset());
            }
            return "";
        }
        return null;
    }

    public String toHl7StringAndReset(String charsetName) {
        String answer = this.toHl7String(charsetName);
        this.reset();
        return answer;
    }

    public synchronized String toPrintFriendlyHl7String() {
        if (this.hasCompleteEnvelope()) {
            int startPosition = this.hasStartOfBlock() ? this.startOfBlockIndex + 1 : 1;
            int endPosition = this.hasEndOfBlock() ? this.endOfBlockIndex : this.availableByteCount - 1;
            return Hl7Util.convertToPrintFriendlyString(this.buffer, startPosition, endPosition);
        }
        return "";
    }

    public String toPrintFriendlyHl7StringAndReset() {
        String answer = this.toPrintFriendlyHl7String();
        this.reset();
        return answer;
    }

    public synchronized byte[] toMllpPayload() {
        byte[] mllpPayload = null;
        if (this.hasCompleteEnvelope()) {
            int length;
            int offset = this.hasStartOfBlock() ? this.startOfBlockIndex + 1 : 1;
            int n = length = this.hasEndOfBlock() ? this.endOfBlockIndex - offset : this.availableByteCount - this.startOfBlockIndex - 1;
            if (length > 0) {
                mllpPayload = new byte[length];
                System.arraycopy(this.buffer, offset, mllpPayload, 0, length);
            } else {
                mllpPayload = new byte[]{};
            }
        }
        return mllpPayload;
    }

    public byte[] toMllpPayloadAndReset() {
        byte[] answer = this.toMllpPayload();
        this.reset();
        return answer;
    }

    public synchronized int getMllpPayloadLength() {
        int answer = -1;
        if (this.hasCompleteEnvelope()) {
            answer = this.isEndOfDataRequired() ? this.endOfBlockIndex - this.startOfBlockIndex + 2 : this.endOfBlockIndex - this.startOfBlockIndex + 2;
        }
        return answer;
    }

    public synchronized int getStartOfBlockIndex() {
        return this.startOfBlockIndex;
    }

    public synchronized int getEndOfBlockIndex() {
        return this.endOfBlockIndex;
    }

    public synchronized boolean hasCompleteEnvelope() {
        if (this.hasStartOfBlock()) {
            if (this.isEndOfDataRequired()) {
                return this.hasEndOfData();
            }
            return this.hasEndOfBlock();
        }
        return false;
    }

    public synchronized boolean hasStartOfBlock() {
        return this.startOfBlockIndex >= 0;
    }

    public synchronized boolean hasEndOfBlock() {
        return this.endOfBlockIndex >= 0;
    }

    public synchronized boolean hasEndOfData() {
        int potentialEndOfDataIndex;
        return this.hasEndOfBlock() && (potentialEndOfDataIndex = this.endOfBlockIndex + 1) < this.availableByteCount && this.buffer[potentialEndOfDataIndex] == 13;
    }

    public synchronized boolean hasOutOfBandData() {
        return this.hasLeadingOutOfBandData() || this.hasTrailingOutOfBandData();
    }

    public synchronized boolean hasLeadingOutOfBandData() {
        return this.size() > 0 && (!this.hasStartOfBlock() || this.startOfBlockIndex > 0);
    }

    public synchronized boolean hasTrailingOutOfBandData() {
        return this.size() > 0 && (this.hasEndOfData() ? this.endOfBlockIndex + 1 < this.size() - 1 : !this.isEndOfDataRequired() && this.hasEndOfBlock() && this.endOfBlockIndex < this.size() - 1);
    }

    public synchronized byte[] getLeadingOutOfBandData() {
        byte[] outOfBandData = null;
        if (this.hasLeadingOutOfBandData()) {
            outOfBandData = new byte[this.startOfBlockIndex == -1 ? this.availableByteCount : this.startOfBlockIndex];
            System.arraycopy(this.buffer, 0, outOfBandData, 0, outOfBandData.length);
        }
        return outOfBandData;
    }

    public synchronized byte[] getTrailingOutOfBandData() {
        byte[] outOfBandData = null;
        if (this.hasTrailingOutOfBandData()) {
            int offset = this.hasEndOfData() ? this.endOfBlockIndex + 2 : this.endOfBlockIndex + 1;
            int length = this.size() - offset;
            outOfBandData = new byte[length];
            System.arraycopy(this.buffer, offset, outOfBandData, 0, length);
        }
        return outOfBandData;
    }

    public synchronized int size() {
        return this.availableByteCount;
    }

    public synchronized int capacity() {
        if (this.buffer != null) {
            return this.buffer.length - this.availableByteCount;
        }
        return -1;
    }

    public synchronized int bufferSize() {
        if (this.buffer != null) {
            return this.buffer.length;
        }
        return -1;
    }

    public byte[] getBuffer() {
        return this.buffer;
    }

    void ensureCapacity(int requiredAvailableCapacity) {
        int currentAvailableCapacity = this.capacity();
        if (requiredAvailableCapacity > currentAvailableCapacity) {
            int requiredBufferSize = this.buffer.length + (requiredAvailableCapacity - currentAvailableCapacity);
            if (this.buffer.length >= 0x40000000) {
                String exceptionMessageFormat = "Cannot increase the buffer size from <%d> to <%d> in order to increase the available capacity from <%d> to <%d> because the buffer is already the maximum size <%d>";
                throw new IllegalStateException(String.format("Cannot increase the buffer size from <%d> to <%d> in order to increase the available capacity from <%d> to <%d> because the buffer is already the maximum size <%d>", this.buffer.length, requiredBufferSize, currentAvailableCapacity, requiredAvailableCapacity, 0x40000000));
            }
            if (requiredBufferSize > 0x40000000) {
                String exceptionMessageFormat = "Cannot increase the buffer size <%d> in order to increase the available capacity from <%d> to <%d> because the required buffer size <%d> exceeds the maximum buffer size <%d>";
                throw new IllegalStateException(String.format("Cannot increase the buffer size <%d> in order to increase the available capacity from <%d> to <%d> because the required buffer size <%d> exceeds the maximum buffer size <%d>", this.buffer.length, currentAvailableCapacity, requiredAvailableCapacity, requiredBufferSize, 0x40000000));
            }
            int newBufferSize = Math.min(this.buffer.length + Math.max(2048, requiredAvailableCapacity), 0x40000000);
            this.buffer = Arrays.copyOf(this.buffer, newBufferSize);
        }
    }

    void updateIndexes(int b, int indexOffset) {
        if (this.startOfBlockIndex < 0) {
            if (b == 11) {
                this.startOfBlockIndex = this.availableByteCount + indexOffset;
            }
        } else if (this.endOfBlockIndex < 0 && b == 28) {
            this.endOfBlockIndex = this.availableByteCount + indexOffset;
        }
    }

    void readSocketInputStream(InputStream socketInputStream, Socket socket) throws MllpSocketException, SocketTimeoutException {
        this.log.trace("readSocketInputStream(socketInputStream, {}) - entering with initial buffer size = {}", (Object)socket, (Object)this.size());
        try {
            int readCount = socketInputStream.read(this.buffer, this.availableByteCount, this.buffer.length - this.availableByteCount);
            if (readCount == -1) {
                String exceptionMessage = String.format("readSocketInputStream(socketInputStream, %s) - END_OF_STREAM returned from SocketInputStream.read(byte[%d], %d, %d)", socket, this.buffer.length, this.availableByteCount, this.buffer.length - this.availableByteCount);
                this.resetSocket(socket);
                throw new MllpSocketException(exceptionMessage);
            }
            if (readCount > 0) {
                for (int i = 0; (this.startOfBlockIndex == -1 || this.endOfBlockIndex == -1) && i < readCount; ++i) {
                    this.updateIndexes(this.buffer[this.availableByteCount + i], i);
                }
                this.availableByteCount += readCount;
                if (this.hasStartOfBlock()) {
                    this.log.trace("readSocketInputStream(socketInputStream, {}) - read {} bytes for a total of {} bytes", new Object[]{socket, readCount, this.availableByteCount});
                } else {
                    this.log.warn("readSocketInputStream(socketInputStream, {}) - ignoring {} bytes received before START_OF_BLOCK", new Object[]{socket, this.size(), this.toPrintFriendlyStringAndReset()});
                }
            }
        }
        catch (SocketTimeoutException timeoutEx) {
            throw timeoutEx;
        }
        catch (IOException ioEx) {
            String exceptionMessage = String.format("readSocketInputStream(socketInputStream, %s) - IOException thrown from SocketInputStream.read(byte[%d], %d, %d) from %s", socket, this.buffer.length, this.availableByteCount, this.buffer.length - this.availableByteCount, socket);
            this.resetSocket(socket);
            throw new MllpSocketException(exceptionMessage, ioEx);
        }
        finally {
            this.log.trace("readSocketInputStream(socketInputStream, {}) - exiting with buffer size = {}", (Object)socket, (Object)this.size());
        }
    }

    public void closeSocket(Socket socket) {
        this.doSocketClose(socket, null, false);
    }

    public void closeSocket(Socket socket, String logMessage) {
        this.doSocketClose(socket, logMessage, false);
    }

    public void resetSocket(Socket socket) {
        this.doSocketClose(socket, null, true);
    }

    public void resetSocket(Socket socket, String logMessage) {
        this.doSocketClose(socket, logMessage, true);
    }

    void doSocketClose(Socket socket, String logMessage, boolean reset) {
        if (socket != null && socket.isConnected() && !socket.isClosed()) {
            if (logMessage != null && !logMessage.isEmpty()) {
                this.log.info("{} - {} socket {}", new Object[]{reset ? "Resetting" : "Closing", logMessage, socket});
            } else {
                this.log.debug("{} socket {}", (Object)(reset ? "Resetting" : "Closing"), (Object)socket);
            }
            this.endpoint.updateLastConnectionTerminatedTicks();
            if (!socket.isInputShutdown()) {
                try {
                    socket.shutdownInput();
                }
                catch (IOException ignoredEx) {
                    this.log.trace("doSocketClose(socket[{}], logMessage[{}], reset[{}] - ignoring exception raised by Socket.shutdownInput()", new Object[]{socket, logMessage, reset, ignoredEx});
                }
            }
            if (!socket.isOutputShutdown()) {
                try {
                    socket.shutdownOutput();
                }
                catch (IOException ignoredEx) {
                    this.log.trace("doSocketClose(socket[{}], logMessage[{}], reset[{}] - ignoring exception raised by Socket.shutdownOutput()", new Object[]{socket, logMessage, reset, ignoredEx});
                }
            }
            if (reset) {
                boolean on = true;
                boolean linger = false;
                try {
                    socket.setSoLinger(true, 0);
                }
                catch (IOException ignoredEx) {
                    this.log.trace("doSocketClose(socket[{}], logMessage[{}], reset[{}] - ignoring exception raised by Socket.setSoLinger({}, {})", new Object[]{socket, logMessage, reset, true, 0, ignoredEx});
                }
            }
            try {
                socket.close();
            }
            catch (IOException ignoredEx) {
                this.log.trace("doSocketClose(socket[{}], logMessage[{}], reset[{}] - ignoring exception raised by Socket.close()", new Object[]{socket, logMessage, reset, ignoredEx});
            }
        }
    }

    public static boolean isConnectionValid(Socket socket) {
        return socket != null && socket.isConnected() && !socket.isClosed();
    }

    public static String formatAddressString(SocketAddress sourceAddress, SocketAddress targetAddress) {
        String sourceAddressString = null;
        String targetAddressString = null;
        if (sourceAddress != null) {
            sourceAddressString = sourceAddress.toString();
        }
        if (targetAddress != null) {
            targetAddressString = targetAddress.toString();
        }
        return String.format("%s => %s", sourceAddressString, targetAddressString);
    }
}

