/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.processor;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.camel.ContextTestSupport;
import org.apache.camel.Expression;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.mock.MockEndpoint;
import org.apache.camel.processor.ThrottlerRejectedExecutionException;

public class ThrottlerTest
extends ContextTestSupport {
    private static final int INTERVAL = 500;
    private static final int TOLERANCE = 50;
    private static final int MESSAGE_COUNT = 9;

    protected boolean canTest() {
        return !ThrottlerTest.isPlatform("windows");
    }

    public void testSendLotsOfMessagesButOnly3GetThroughWithin2Seconds() throws Exception {
        if (!this.canTest()) {
            return;
        }
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        resultEndpoint.expectedMessageCount(3);
        resultEndpoint.setResultWaitTime(2000L);
        for (int i = 0; i < 9; ++i) {
            this.template.sendBody("seda:a", (Object)("<message>" + i + "</message>"));
        }
        resultEndpoint.assertIsSatisfied();
    }

    public void testSendLotsOfMessagesWithRejectExecution() throws Exception {
        if (!this.canTest()) {
            return;
        }
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        resultEndpoint.expectedMessageCount(2);
        resultEndpoint.setResultWaitTime(2000L);
        MockEndpoint errorEndpoint = this.resolveMandatoryEndpoint("mock:error", MockEndpoint.class);
        errorEndpoint.expectedMessageCount(4);
        errorEndpoint.setResultWaitTime(2000L);
        for (int i = 0; i < 6; ++i) {
            this.template.sendBody("direct:start", (Object)("<message>" + i + "</message>"));
        }
        resultEndpoint.assertIsSatisfied();
        errorEndpoint.assertIsSatisfied();
    }

    public void testSendLotsOfMessagesSimultaneouslyButOnly3GetThrough() throws Exception {
        if (!this.canTest()) {
            return;
        }
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        long elapsed = this.sendMessagesAndAwaitDelivery(9, "direct:a", 9, resultEndpoint);
        this.assertThrottlerTiming(elapsed, 1, 500, 9);
    }

    public void testConfigurationWithConstantExpression() throws Exception {
        if (!this.canTest()) {
            return;
        }
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        long elapsed = this.sendMessagesAndAwaitDelivery(9, "direct:expressionConstant", 9, resultEndpoint);
        this.assertThrottlerTiming(elapsed, 1, 500, 9);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testConfigurationWithHeaderExpression() throws Exception {
        if (!this.canTest()) {
            return;
        }
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        resultEndpoint.expectedMessageCount(9);
        ExecutorService executor = Executors.newFixedThreadPool(9);
        try {
            this.sendMessagesWithHeaderExpression(executor, resultEndpoint, 1, 500, 9);
        }
        finally {
            executor.shutdownNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testConfigurationWithChangingHeaderExpression() throws Exception {
        if (!this.canTest()) {
            return;
        }
        ExecutorService executor = Executors.newFixedThreadPool(5);
        try {
            MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
            this.sendMessagesWithHeaderExpression(executor, resultEndpoint, 1, 500, 9);
            Thread.sleep(550L);
            resultEndpoint.reset();
            this.sendMessagesWithHeaderExpression(executor, resultEndpoint, 10, 500, 9);
            Thread.sleep(550L);
            resultEndpoint.reset();
            this.sendMessagesWithHeaderExpression(executor, resultEndpoint, 1, 500, 9);
            Thread.sleep(550L);
            resultEndpoint.reset();
            this.sendMessagesWithHeaderExpression(executor, resultEndpoint, 10, 500, 9);
        }
        finally {
            executor.shutdownNow();
        }
    }

    public void testHighThrottleRate() throws Exception {
        if (!this.canTest()) {
            return;
        }
        int messageCount = 20000;
        MockEndpoint resultEndpoint = this.resolveMandatoryEndpoint("mock:result", MockEndpoint.class);
        long elapsed = this.sendMessagesAndAwaitDelivery(messageCount, "direct:highThrottleRate", 5, resultEndpoint);
        this.assertThrottlerTiming(elapsed, 1000, 500, messageCount);
    }

    private void assertThrottlerTiming(long elapsedTimeMs, int throttle, int intervalMs, int messageCount) {
        long minimum = this.calculateMinimum(intervalMs, throttle, messageCount) - 50L;
        long maximum = this.calculateMaximum(intervalMs, throttle, messageCount) + 50L;
        this.log.info("Sent {} exchanges in {}ms, with throttle rate of {} per {}ms. Calculated min {}ms and max {}ms", new Object[]{messageCount, elapsedTimeMs, throttle, intervalMs, minimum, maximum});
        ThrottlerTest.assertTrue((String)("Should take at least " + minimum + "ms, was: " + elapsedTimeMs), (elapsedTimeMs >= minimum ? 1 : 0) != 0);
        ThrottlerTest.assertTrue((String)("Should take at most " + maximum + "ms, was: " + elapsedTimeMs), (elapsedTimeMs <= maximum + 50L ? 1 : 0) != 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long sendMessagesAndAwaitDelivery(int messageCount, final String endpointUri, int threadPoolSize, MockEndpoint receivingEndpoint) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(threadPoolSize);
        try {
            if (receivingEndpoint != null) {
                receivingEndpoint.expectedMessageCount(messageCount);
            }
            long start = System.nanoTime();
            for (int i = 0; i < messageCount; ++i) {
                executor.execute(new Runnable(){

                    @Override
                    public void run() {
                        ThrottlerTest.this.template.sendBody(endpointUri, (Object)"<message>payload</message>");
                    }
                });
            }
            if (receivingEndpoint != null) {
                receivingEndpoint.assertIsSatisfied();
            }
            long l = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start);
            return l;
        }
        finally {
            executor.shutdownNow();
        }
    }

    private void sendMessagesWithHeaderExpression(ExecutorService executor, MockEndpoint resultEndpoint, final int throttle, int intervalMs, int messageCount) throws InterruptedException {
        resultEndpoint.expectedMessageCount(messageCount);
        long start = System.nanoTime();
        for (int i = 0; i < messageCount; ++i) {
            executor.execute(new Runnable(){

                @Override
                public void run() {
                    ThrottlerTest.this.template.sendBodyAndHeader("direct:expressionHeader", (Object)"<message>payload</message>", "throttleValue", (Object)throttle);
                }
            });
        }
        resultEndpoint.assertIsSatisfied();
        long elapsed = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start);
        this.assertThrottlerTiming(elapsed, throttle, intervalMs, messageCount);
    }

    private long calculateMinimum(long periodMs, long throttleRate, long messageCount) {
        if (messageCount % throttleRate > 0L) {
            return (long)Math.floor((double)messageCount / (double)throttleRate) * periodMs;
        }
        return (long)(Math.floor((double)messageCount / (double)throttleRate) * (double)periodMs) - periodMs;
    }

    private long calculateMaximum(long periodMs, long throttleRate, long messageCount) {
        return (long)Math.ceil((double)messageCount / (double)throttleRate) * periodMs;
    }

    @Override
    protected RouteBuilder createRouteBuilder() {
        return new RouteBuilder(){

            public void configure() {
                this.onException(ThrottlerRejectedExecutionException.class).handled(true).to("mock:error");
                this.from("seda:a").throttle(3L).timePeriodMillis(10000L).to(new String[]{"log:result", "mock:result"});
                this.from("direct:a").throttle(1L).timePeriodMillis(500L).to(new String[]{"log:result", "mock:result"});
                this.from("direct:expressionConstant").throttle((Expression)this.constant(1)).timePeriodMillis(500L).to(new String[]{"log:result", "mock:result"});
                this.from("direct:expressionHeader").throttle((Expression)this.header("throttleValue")).timePeriodMillis(500L).to(new String[]{"log:result", "mock:result"});
                this.from("direct:start").throttle(2L).timePeriodMillis(10000L).rejectExecution(true).to(new String[]{"log:result", "mock:result"});
                this.from("direct:highThrottleRate").throttle(1000L).timePeriodMillis(500L).to("mock:result");
            }
        };
    }
}

