/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.beam.sdk.schemas.logicaltypes;

import static org.junit.Assert.assertEquals;

import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.UUID;
import org.apache.beam.sdk.schemas.Schema;
import org.apache.beam.sdk.schemas.Schema.Field;
import org.apache.beam.sdk.schemas.Schema.FieldType;
import org.apache.beam.sdk.schemas.logicaltypes.OneOfType.Value;
import org.apache.beam.sdk.values.Row;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableMap;
import org.junit.Test;

/** Unit tests for logical types. */
public class LogicalTypesTest {
  @Test
  public void testEnumeration() {
    Map<String, Integer> enumMap = ImmutableMap.of("FIRST", 1, "SECOND", 2);
    EnumerationType enumeration = EnumerationType.create(enumMap);
    assertEquals(enumeration.valueOf(1), enumeration.valueOf("FIRST"));
    assertEquals(enumeration.valueOf(2), enumeration.valueOf("SECOND"));
    assertEquals("FIRST", enumeration.toString(enumeration.valueOf(1)));
    assertEquals(1, enumeration.valueOf("FIRST").getValue());
    assertEquals("SECOND", enumeration.toString(enumeration.valueOf(2)));
    assertEquals(2, enumeration.valueOf("SECOND").getValue());

    Schema schema =
        Schema.builder().addLogicalTypeField("enum", EnumerationType.create(enumMap)).build();
    Row row1 = Row.withSchema(schema).addValue(enumeration.valueOf(1)).build();
    Row row2 = Row.withSchema(schema).addValue(enumeration.valueOf("FIRST")).build();
    assertEquals(row1, row2);
    assertEquals(1, row1.getLogicalTypeValue(0, EnumerationType.Value.class).getValue());

    Row row3 = Row.withSchema(schema).addValue(enumeration.valueOf(2)).build();
    Row row4 = Row.withSchema(schema).addValue(enumeration.valueOf("SECOND")).build();
    assertEquals(row3, row4);
    assertEquals(2, row3.getLogicalTypeValue(0, EnumerationType.Value.class).getValue());
  }

  @Test
  public void testOneOf() {
    OneOfType oneOf =
        OneOfType.create(Field.of("string", FieldType.STRING), Field.of("int32", FieldType.INT32));
    Schema schema = Schema.builder().addLogicalTypeField("union", oneOf).build();

    Row stringOneOf =
        Row.withSchema(schema).addValue(oneOf.createValue("string", "stringValue")).build();
    Value union = stringOneOf.getLogicalTypeValue(0, OneOfType.Value.class);
    assertEquals("string", oneOf.getCaseEnumType().toString(union.getCaseType()));
    assertEquals("stringValue", union.getValue());

    Row intOneOf = Row.withSchema(schema).addValue(oneOf.createValue("int32", 42)).build();
    union = intOneOf.getLogicalTypeValue(0, OneOfType.Value.class);
    assertEquals("int32", oneOf.getCaseEnumType().toString(union.getCaseType()));
    assertEquals(42, (int) union.getValue());
  }

  @Test
  public void testNanosInstant() {
    Schema rowSchema = new NanosInstant().getBaseType().getRowSchema();
    Instant now = Instant.now();
    Row nowAsRow = Row.withSchema(rowSchema).addValues(now.getEpochSecond(), now.getNano()).build();

    Schema schema = Schema.builder().addLogicalTypeField("now", new NanosInstant()).build();
    Row row = Row.withSchema(schema).addValues(now).build();
    assertEquals(now, row.getLogicalTypeValue(0, NanosInstant.class));
    assertEquals(nowAsRow, row.getBaseValue(0, Row.class));
  }

  @Test
  public void testNanosDuration() {
    Schema rowSchema = new NanosInstant().getBaseType().getRowSchema();
    Duration duration = Duration.ofSeconds(123, 42);
    Row durationAsRow = Row.withSchema(rowSchema).addValues(123L, 42).build();

    Schema schema = Schema.builder().addLogicalTypeField("duration", new NanosDuration()).build();
    Row row = Row.withSchema(schema).addValues(duration).build();
    assertEquals(duration, row.getLogicalTypeValue(0, NanosDuration.class));
    assertEquals(durationAsRow, row.getBaseValue(0, Row.class));
  }

  @Test
  public void testUuid() {
    Schema rowSchema = new UuidLogicalType().getBaseType().getRowSchema();
    UUID uuid = UUID.randomUUID();
    Row uuidAsRow =
        Row.withSchema(rowSchema)
            .addValues(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits())
            .build();

    Schema schema = Schema.builder().addLogicalTypeField("uuid", new UuidLogicalType()).build();
    Row row = Row.withSchema(schema).addValues(uuid).build();
    assertEquals(uuid, row.getLogicalTypeValue(0, UUID.class));
    assertEquals(uuidAsRow, row.getBaseValue(0, Row.class));
  }
}
