/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.beam.runners.dataflow.worker.util.common.worker;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

/** Unit tests for {@link ShuffleEntry}. */
@RunWith(JUnit4.class)
public class ShuffleEntryTest {
  private static final byte[] KEY = {0xA};
  private static final byte[] SKEY = {0xB};
  private static final byte[] VALUE = {0xC};

  @Test
  public void accessors() {
    ShuffleEntry entry = new ShuffleEntry(KEY, SKEY, VALUE);
    assertThat(entry.getKey(), equalTo(KEY));
    assertThat(entry.getSecondaryKey(), equalTo(SKEY));
    assertThat(entry.getValue(), equalTo(VALUE));
  }

  @Test
  @SuppressWarnings("SelfEquals")
  public void equalsToItself() {
    ShuffleEntry entry = new ShuffleEntry(KEY, SKEY, VALUE);
    assertTrue(entry.equals(entry));
  }

  @Test
  public void equalsForEqualEntries() {
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(KEY.clone(), SKEY.clone(), VALUE.clone());

    assertTrue(entry0.equals(entry1));
    assertTrue(entry1.equals(entry0));
    assertEquals(entry0.hashCode(), entry1.hashCode());
  }

  @Test
  public void equalsForEqualNullEntries() {
    ShuffleEntry entry0 = new ShuffleEntry(null, null, null);
    ShuffleEntry entry1 = new ShuffleEntry(null, null, null);

    assertTrue(entry0.equals(entry1));
    assertTrue(entry1.equals(entry0));
    assertEquals(entry0.hashCode(), entry1.hashCode());
  }

  @Test
  public void notEqualsWhenKeysDiffer() {
    final byte[] otherKey = {0x1};
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(otherKey, SKEY, VALUE);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void notEqualsWhenKeysDifferOneNull() {
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(null, SKEY, VALUE);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void notEqualsWhenSecondaryKeysDiffer() {
    final byte[] otherSKey = {0x2};
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(KEY, otherSKey, VALUE);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void notEqualsWhenSecondaryKeysDifferOneNull() {
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(KEY, null, VALUE);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void notEqualsWhenValuesDiffer() {
    final byte[] otherValue = {0x2};
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(KEY, SKEY, otherValue);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void notEqualsWhenValuesDifferOneNull() {
    ShuffleEntry entry0 = new ShuffleEntry(KEY, SKEY, VALUE);
    ShuffleEntry entry1 = new ShuffleEntry(KEY, SKEY, null);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }

  @Test
  public void emptyNotTheSameAsNull() {
    final byte[] empty = {};
    ShuffleEntry entry0 = new ShuffleEntry(null, null, null);
    ShuffleEntry entry1 = new ShuffleEntry(empty, empty, empty);

    assertFalse(entry0.equals(entry1));
    assertFalse(entry1.equals(entry0));
    assertThat(entry0.hashCode(), not(equalTo(entry1.hashCode())));
  }
}
