/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.beam.examples.complete;

import java.util.Date;
import org.apache.beam.sdk.io.FileSystems;
import org.apache.beam.sdk.io.fs.ResolveOptions.StandardResolveOptions;
import org.apache.beam.sdk.options.PipelineOptionsFactory;
import org.apache.beam.sdk.testing.FileChecksumMatcher;
import org.apache.beam.sdk.testing.TestPipeline;
import org.apache.beam.sdk.testing.TestPipelineOptions;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

/** End-to-end tests of TopWikipediaSessions. */
@RunWith(JUnit4.class)
public class TopWikipediaSessionsIT {

  private static final String DEFAULT_INPUT_10_FILES =
      "gs://apache-beam-samples/wikipedia_edits/wiki_data-00000000000*.json";
  private static final String DEFAULT_OUTPUT_CHECKSUM = "61262b08503338bfe4e36b0791958d65e6070933";

  /** PipelineOptions for the TopWikipediaSessions integration test. */
  public interface TopWikipediaSessionsITOptions
      extends TestPipelineOptions, TopWikipediaSessions.Options {}

  @BeforeClass
  public static void setUp() {
    PipelineOptionsFactory.register(TestPipelineOptions.class);
  }

  @Test
  public void testE2ETopWikiPages() throws Exception {
    TopWikipediaSessionsITOptions options =
        TestPipeline.testingPipelineOptions().as(TopWikipediaSessionsITOptions.class);

    options.setWikiInput(DEFAULT_INPUT_10_FILES);
    options.setOutput(
        FileSystems.matchNewResource(options.getTempRoot(), true)
            .resolve(
                String.format("topwikisessions-it-%tF-%<tH-%<tM-%<tS-%<tL", new Date()),
                StandardResolveOptions.RESOLVE_DIRECTORY)
            .resolve("output", StandardResolveOptions.RESOLVE_DIRECTORY)
            .resolve("results", StandardResolveOptions.RESOLVE_FILE)
            .toString());
    options.setOnSuccessMatcher(
        new FileChecksumMatcher(DEFAULT_OUTPUT_CHECKSUM, options.getOutput() + "*-of-*"));

    TopWikipediaSessions.run(options);
  }
}
