/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.beam.sdk.nexmark.queries.sql;

import static org.apache.beam.sdk.nexmark.model.sql.adapter.ModelAdaptersMapping.ADAPTERS;
import static org.junit.Assert.assertEquals;

import org.apache.beam.sdk.nexmark.model.Bid;
import org.apache.beam.sdk.nexmark.model.Event;
import org.apache.beam.sdk.nexmark.model.sql.adapter.ModelFieldsAdapter;
import org.apache.beam.sdk.testing.PAssert;
import org.apache.beam.sdk.testing.TestPipeline;
import org.apache.beam.sdk.testing.TestStream;
import org.apache.beam.sdk.values.PCollection;
import org.apache.beam.sdk.values.Row;
import org.junit.Rule;
import org.junit.Test;

/**
 * Unit tests for {@link SqlQuery1}.
 */
public class SqlQuery1Test {

  private static final Bid BID1_USD =
      new Bid(5L, 3L, 100L, 43234234L, "extra1");

  private static final Bid BID2_USD =
      new Bid(6L, 4L, 500L, 13234234L, "extra2");

  private static final Bid BID1_EUR =
      new Bid(5L, 3L, 89L, 43234234L, "extra1");

  private static final Bid BID2_EUR =
      new Bid(6L, 4L, 445L, 13234234L, "extra2");

  private static final ModelFieldsAdapter<Bid> BID_ADAPTER =
      ADAPTERS.get(Bid.class);

  private static final Row BID1_EUR_ROW =
      Row
          .withRowType(BID_ADAPTER.getRowType())
          .addValues(BID_ADAPTER.getFieldsValues(BID1_EUR))
          .build();

  private static final Row BID2_EUR_ROW =
      Row
          .withRowType(BID_ADAPTER.getRowType())
          .addValues(BID_ADAPTER.getFieldsValues(BID2_EUR))
          .build();

  @Rule
  public TestPipeline testPipeline = TestPipeline.create();

  @Test
  public void testDolToEurConversion() {
    SqlQuery1.DolToEur dolToEur = new SqlQuery1.DolToEur();
    assertEquals(Long.valueOf(445), dolToEur.apply(500L));
  }

  @Test
  public void testConvertsPriceToEur() throws Exception {
    PCollection<Event> bids = testPipeline.apply(
        TestStream.create(Event.CODER)
            .addElements(new Event(BID1_USD))
            .addElements(new Event(BID2_USD))
            .advanceWatermarkToInfinity());

    PAssert
        .that(bids.apply(new SqlQuery1()))
        .containsInAnyOrder(BID1_EUR_ROW, BID2_EUR_ROW);

    testPipeline.run();
  }
}
