/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.beam.runners.flink;

import org.apache.beam.model.pipeline.v1.RunnerApi;
import org.apache.beam.runners.core.construction.PipelineOptionsTranslation;
import org.apache.beam.runners.core.construction.PipelineTranslation;
import org.apache.beam.sdk.Pipeline;
import org.apache.beam.sdk.io.GenerateSequence;
import org.apache.beam.sdk.options.PipelineOptionsFactory;
import org.apache.beam.vendor.grpc.v1p21p0.com.google.protobuf.Struct;
import org.apache.flink.client.program.OptimizerPlanEnvironment;
import org.apache.flink.client.program.PackagedProgram;
import org.apache.flink.client.program.ProgramInvocationException;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.optimizer.Optimizer;
import org.hamcrest.CoreMatchers;
import org.hamcrest.MatcherAssert;
import org.hamcrest.core.StringContains;
import org.junit.Assert;
import org.junit.Test;

/** Tests for {@link FlinkJobInvoker}. */
public class FlinkJobInvokerTest {

  @Test
  public void testEnsureStdoutStdErrIsRestored() throws Exception {
    PackagedProgram packagedProgram = new PackagedProgram(getClass());
    OptimizerPlanEnvironment env = new OptimizerPlanEnvironment(new Optimizer(new Configuration()));
    try {
      // Flink will throw an error because no job graph will be generated by the main method
      env.getOptimizedPlan(packagedProgram);
      Assert.fail("This should have failed to create the Flink Plan.");
    } catch (ProgramInvocationException e) {
      // Test that Flink wasn't able to intercept the stdout/stderr and we printed to the regular
      // output instead
      MatcherAssert.assertThat(
          e.getMessage(),
          CoreMatchers.allOf(
              StringContains.containsString("System.out: (none)"),
              StringContains.containsString("System.err: (none)")));
    }
  }

  /** Main method for {@code testEnsureStdoutStdErrIsRestored()}. */
  public static void main(String[] args) {
    Pipeline p = Pipeline.create();
    p.apply(GenerateSequence.from(0));

    RunnerApi.Pipeline pipeline = PipelineTranslation.toProto(p);
    Struct options = PipelineOptionsTranslation.toProto(PipelineOptionsFactory.create());

    FlinkJobInvoker flinkJobInvoker =
        FlinkJobInvoker.create(new FlinkJobServerDriver.FlinkServerConfiguration());
    // This will call Workarounds.restoreOriginalStdOutAndStdErr() which we want to test
    flinkJobInvoker.invokeWithExecutor(pipeline, options, "retrievalToken", null);
  }
}
