/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.beam.runners.spark;

import javax.annotation.Nullable;
import org.apache.beam.sdk.options.Default;
import org.apache.beam.sdk.options.DefaultValueFactory;
import org.apache.beam.sdk.options.Description;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.testing.TestPipelineOptions;
import org.apache.beam.sdk.transforms.windowing.BoundedWindow;

/** A {@link SparkPipelineOptions} for tests. */
public interface TestSparkPipelineOptions extends SparkPipelineOptions, TestPipelineOptions {

  @Description("A special flag that forces streaming in tests.")
  @Default.Boolean(false)
  boolean isForceStreaming();

  void setForceStreaming(boolean forceStreaming);

  @Description("A hard-coded expected number of assertions for this test pipeline.")
  @Nullable
  Integer getExpectedAssertions();

  void setExpectedAssertions(Integer expectedAssertions);

  @Description(
      "A watermark (time in millis) that causes a pipeline that reads "
          + "from an unbounded source to stop.")
  @Default.InstanceFactory(DefaultStopPipelineWatermarkFactory.class)
  Long getStopPipelineWatermark();

  void setStopPipelineWatermark(Long stopPipelineWatermark);

  /**
   * A factory to provide the default watermark to stop a pipeline that reads from an unbounded
   * source.
   */
  class DefaultStopPipelineWatermarkFactory implements DefaultValueFactory<Long> {
    @Override
    public Long create(PipelineOptions options) {
      return BoundedWindow.TIMESTAMP_MAX_VALUE.getMillis();
    }
  }
}
