/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.processor;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.camel.AsyncCallback;
import org.apache.camel.CamelContext;
import org.apache.camel.Exchange;
import org.apache.camel.Expression;
import org.apache.camel.RuntimeExchangeException;
import org.apache.camel.Traceable;
import org.apache.camel.processor.ThrottlerRejectedExecutionException;
import org.apache.camel.spi.IdAware;
import org.apache.camel.spi.RouteIdAware;
import org.apache.camel.support.AsyncProcessorSupport;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Throttler
extends AsyncProcessorSupport
implements Traceable,
IdAware,
RouteIdAware {
    private static final Logger LOG = LoggerFactory.getLogger(Throttler.class);
    private static final String DEFAULT_KEY = "CamelThrottlerDefaultKey";
    private static final String PROPERTY_EXCHANGE_QUEUED_TIMESTAMP = "CamelThrottlerExchangeQueuedTimestamp";
    private static final String PROPERTY_EXCHANGE_STATE = "CamelThrottlerExchangeState";
    private final CamelContext camelContext;
    private final ScheduledExecutorService asyncExecutor;
    private final boolean shutdownAsyncExecutor;
    private volatile long timePeriodMillis;
    private final long cleanPeriodMillis;
    private String id;
    private String routeId;
    private Expression maxRequestsPerPeriodExpression;
    private boolean rejectExecution;
    private boolean asyncDelayed;
    private boolean callerRunsWhenRejected = true;
    private final Expression correlationExpression;
    private final Map<String, ThrottlingState> states = new ConcurrentHashMap<String, ThrottlingState>();

    public Throttler(CamelContext camelContext, Expression maxRequestsPerPeriodExpression, long timePeriodMillis, ScheduledExecutorService asyncExecutor, boolean shutdownAsyncExecutor, boolean rejectExecution, Expression correlation) {
        this.camelContext = camelContext;
        this.rejectExecution = rejectExecution;
        this.shutdownAsyncExecutor = shutdownAsyncExecutor;
        ObjectHelper.notNull((Object)maxRequestsPerPeriodExpression, (String)"maxRequestsPerPeriodExpression");
        this.maxRequestsPerPeriodExpression = maxRequestsPerPeriodExpression;
        if (timePeriodMillis <= 0L) {
            throw new IllegalArgumentException("TimePeriodMillis should be a positive number, was: " + timePeriodMillis);
        }
        this.timePeriodMillis = timePeriodMillis;
        this.cleanPeriodMillis = timePeriodMillis * 10L;
        this.asyncExecutor = asyncExecutor;
        this.correlationExpression = correlation;
    }

    public boolean process(Exchange exchange, AsyncCallback callback) {
        long queuedStart = 0L;
        if (LOG.isTraceEnabled()) {
            queuedStart = (Long)exchange.getProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP, (Object)0L, Long.class);
            exchange.removeProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP);
        }
        State state = (State)((Object)exchange.getProperty(PROPERTY_EXCHANGE_STATE, (Object)State.SYNC, State.class));
        exchange.removeProperty(PROPERTY_EXCHANGE_STATE);
        boolean doneSync = state == State.SYNC || state == State.ASYNC_REJECTED;
        try {
            if (!this.isRunAllowed()) {
                throw new RejectedExecutionException("Run is not allowed");
            }
            String key = DEFAULT_KEY;
            if (this.correlationExpression != null) {
                key = (String)this.correlationExpression.evaluate(exchange, String.class);
            }
            ThrottlingState throttlingState = this.states.computeIfAbsent(key, x$0 -> new ThrottlingState((String)x$0));
            throttlingState.calculateAndSetMaxRequestsPerPeriod(exchange);
            ThrottlePermit permit = throttlingState.poll();
            if (permit == null) {
                if (this.isRejectExecution()) {
                    throw new ThrottlerRejectedExecutionException("Exceeded the max throttle rate of " + throttlingState.getThrottleRate() + " within " + this.timePeriodMillis + "ms");
                }
                if (this.isAsyncDelayed() && !exchange.isTransacted() && state == State.SYNC) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Throttle rate exceeded but AsyncDelayed enabled, so queueing for async processing, exchangeId: {}", (Object)exchange.getExchangeId());
                    }
                    return this.processAsynchronously(exchange, callback, throttlingState);
                }
                long start = 0L;
                long elapsed = 0L;
                if (LOG.isTraceEnabled()) {
                    start = System.currentTimeMillis();
                }
                permit = throttlingState.take();
                if (LOG.isTraceEnabled()) {
                    elapsed = System.currentTimeMillis() - start;
                }
                throttlingState.enqueue(permit, exchange);
                if (state == State.ASYNC) {
                    if (LOG.isTraceEnabled()) {
                        long queuedTime = start - queuedStart;
                        if (LOG.isTraceEnabled()) {
                            LOG.trace("Queued for {}ms, Throttled for {}ms, exchangeId: {}", new Object[]{queuedTime, elapsed, exchange.getExchangeId()});
                        }
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace("Throttled for {}ms, exchangeId: {}", (Object)elapsed, (Object)exchange.getExchangeId());
                }
            } else {
                throttlingState.enqueue(permit, exchange);
                if (state == State.ASYNC) {
                    if (LOG.isTraceEnabled()) {
                        long queuedTime = System.currentTimeMillis() - queuedStart;
                        LOG.trace("Queued for {}ms, No throttling applied (throttle cleared while queued), for exchangeId: {}", (Object)queuedTime, (Object)exchange.getExchangeId());
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace("No throttling applied to exchangeId: {}", (Object)exchange.getExchangeId());
                }
            }
            callback.done(doneSync);
            return doneSync;
        }
        catch (InterruptedException e) {
            boolean forceShutdown = exchange.getContext().getShutdownStrategy().isForceShutdown();
            if (forceShutdown) {
                String msg = "Run not allowed as ShutdownStrategy is forcing shutting down, will reject executing exchange: " + exchange;
                LOG.debug(msg);
                exchange.setException((Throwable)new RejectedExecutionException(msg, e));
            } else {
                exchange.setException((Throwable)e);
            }
            callback.done(doneSync);
            return doneSync;
        }
        catch (Throwable t) {
            exchange.setException(t);
            callback.done(doneSync);
            return doneSync;
        }
    }

    protected boolean processAsynchronously(Exchange exchange, AsyncCallback callback, ThrottlingState throttlingState) {
        try {
            if (LOG.isTraceEnabled()) {
                exchange.setProperty(PROPERTY_EXCHANGE_QUEUED_TIMESTAMP, (Object)System.currentTimeMillis());
            }
            exchange.setProperty(PROPERTY_EXCHANGE_STATE, (Object)State.ASYNC);
            long delay = throttlingState.peek().getDelay(TimeUnit.NANOSECONDS);
            this.asyncExecutor.schedule(() -> this.process(exchange, callback), delay, TimeUnit.NANOSECONDS);
            return false;
        }
        catch (RejectedExecutionException e) {
            if (this.isCallerRunsWhenRejected()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("AsyncExecutor is full, rejected exchange will run in the current thread, exchangeId: {}", (Object)exchange.getExchangeId());
                }
                exchange.setProperty(PROPERTY_EXCHANGE_STATE, (Object)State.ASYNC_REJECTED);
                return this.process(exchange, callback);
            }
            throw e;
        }
    }

    protected void doStart() throws Exception {
        if (this.isAsyncDelayed()) {
            ObjectHelper.notNull((Object)this.asyncExecutor, (String)"executorService", (Object)((Object)this));
        }
    }

    protected void doShutdown() throws Exception {
        if (this.shutdownAsyncExecutor && this.asyncExecutor != null) {
            this.camelContext.getExecutorServiceManager().shutdownNow((ExecutorService)this.asyncExecutor);
        }
        this.states.clear();
        super.doShutdown();
    }

    public boolean isRejectExecution() {
        return this.rejectExecution;
    }

    public void setRejectExecution(boolean rejectExecution) {
        this.rejectExecution = rejectExecution;
    }

    public boolean isAsyncDelayed() {
        return this.asyncDelayed;
    }

    public void setAsyncDelayed(boolean asyncDelayed) {
        this.asyncDelayed = asyncDelayed;
    }

    public boolean isCallerRunsWhenRejected() {
        return this.callerRunsWhenRejected;
    }

    public void setCallerRunsWhenRejected(boolean callerRunsWhenRejected) {
        this.callerRunsWhenRejected = callerRunsWhenRejected;
    }

    public String getId() {
        return this.id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public String getRouteId() {
        return this.routeId;
    }

    public void setRouteId(String routeId) {
        this.routeId = routeId;
    }

    public void setMaximumRequestsPerPeriodExpression(Expression maxRequestsPerPeriodExpression) {
        this.maxRequestsPerPeriodExpression = maxRequestsPerPeriodExpression;
    }

    public Expression getMaximumRequestsPerPeriodExpression() {
        return this.maxRequestsPerPeriodExpression;
    }

    public int getCurrentMaximumRequestsPerPeriod() {
        return this.states.values().stream().mapToInt(ThrottlingState::getThrottleRate).max().orElse(0);
    }

    public void setTimePeriodMillis(long timePeriodMillis) {
        this.timePeriodMillis = timePeriodMillis;
    }

    public long getTimePeriodMillis() {
        return this.timePeriodMillis;
    }

    public String getTraceLabel() {
        return "throttle[" + this.maxRequestsPerPeriodExpression + " per: " + this.timePeriodMillis + "]";
    }

    public String toString() {
        return this.id;
    }

    private static enum State {
        SYNC,
        ASYNC,
        ASYNC_REJECTED;

    }

    private class ThrottlingState {
        private final String key;
        private final DelayQueue<ThrottlePermit> delayQueue = new DelayQueue();
        private final AtomicReference<ScheduledFuture<?>> cleanFuture = new AtomicReference();
        private volatile int throttleRate;

        ThrottlingState(String key) {
            this.key = key;
        }

        public int getThrottleRate() {
            return this.throttleRate;
        }

        public ThrottlePermit poll() {
            return (ThrottlePermit)this.delayQueue.poll();
        }

        public ThrottlePermit peek() {
            return (ThrottlePermit)this.delayQueue.peek();
        }

        public ThrottlePermit take() throws InterruptedException {
            return (ThrottlePermit)this.delayQueue.take();
        }

        public void clean() {
            Throttler.this.states.remove(this.key);
        }

        public void enqueue(ThrottlePermit permit, Exchange exchange) {
            permit.setDelayMs(Throttler.this.getTimePeriodMillis());
            this.delayQueue.put(permit);
            try {
                ScheduledFuture<?> next = Throttler.this.asyncExecutor.schedule(this::clean, Throttler.this.cleanPeriodMillis, TimeUnit.MILLISECONDS);
                ScheduledFuture<?> prev = this.cleanFuture.getAndSet(next);
                if (prev != null) {
                    prev.cancel(false);
                }
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Permit released, for exchangeId: {}", (Object)exchange.getExchangeId());
                }
            }
            catch (RejectedExecutionException e) {
                LOG.debug("Throttling queue cleaning rejected", (Throwable)e);
            }
        }

        public synchronized void calculateAndSetMaxRequestsPerPeriod(Exchange exchange) throws Exception {
            Integer newThrottle = (Integer)Throttler.this.maxRequestsPerPeriodExpression.evaluate(exchange, Integer.class);
            if (newThrottle != null && newThrottle < 0) {
                throw new IllegalStateException("The maximumRequestsPerPeriod must be a positive number, was: " + newThrottle);
            }
            if (newThrottle == null && this.throttleRate == 0) {
                throw new RuntimeExchangeException("The maxRequestsPerPeriodExpression was evaluated as null: " + Throttler.this.maxRequestsPerPeriodExpression, exchange);
            }
            if (newThrottle != null && newThrottle != this.throttleRate) {
                if (this.throttleRate > newThrottle) {
                    for (int delta = this.throttleRate - newThrottle; delta > 0; --delta) {
                        this.delayQueue.take();
                        if (!LOG.isTraceEnabled()) continue;
                        LOG.trace("Permit discarded due to throttling rate decrease, triggered by ExchangeId: {}", (Object)exchange.getExchangeId());
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Throttle rate decreased from {} to {}, triggered by ExchangeId: {}", new Object[]{this.throttleRate, newThrottle, exchange.getExchangeId()});
                    }
                } else if (newThrottle > this.throttleRate) {
                    int delta = newThrottle - this.throttleRate;
                    for (int i = 0; i < delta; ++i) {
                        this.delayQueue.put(new ThrottlePermit(-1L));
                    }
                    if (this.throttleRate == 0) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Initial throttle rate set to {}, triggered by ExchangeId: {}", (Object)newThrottle, (Object)exchange.getExchangeId());
                        }
                    } else if (LOG.isDebugEnabled()) {
                        LOG.debug("Throttle rate increase from {} to {}, triggered by ExchangeId: {}", new Object[]{this.throttleRate, newThrottle, exchange.getExchangeId()});
                    }
                }
                this.throttleRate = newThrottle;
            }
        }
    }

    private static class ThrottlePermit
    implements Delayed {
        private volatile long scheduledTime;

        ThrottlePermit(long delayMs) {
            this.setDelayMs(delayMs);
        }

        public void setDelayMs(long delayMs) {
            this.scheduledTime = System.currentTimeMillis() + delayMs;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.scheduledTime - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed o) {
            return Long.compare(this.getDelay(TimeUnit.MILLISECONDS), o.getDelay(TimeUnit.MILLISECONDS));
        }
    }
}

